/**
 * \file bricks.c
 *
 * \brief Implmentation de diverses fonctions gnriques de base
 * \author Denis MARTIN
 * \date 2017-2019
 */


/*

Copyright (C) 2017, 2019 Denis MARTIN

---------------------------------------

This file is part of MinD.

MinD is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or any later version.

MinD is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with MinD.  If not, see <http://www.gnu.org/licenses/>.

--------------------------------------------------------------------------

Ce fichier fait partie de MinD.

MinD est un logiciel libre : il peut tre redistribu et/ou modifi
selon les termes de la Licence Publique Gnrale GNU telle que publie par
la Free Software Foundation, soit la version 3 de la licence ou toute
version ultrieure.

MinD est distribu dans l'espoir qu'il sera utile,
mais SANS AUCUNE GARANTIE ; sans mme la garantie implicite de
VALEUR MARCHANDE ou d'ADEQUATION AVEC UN BUT PARTICULIER. Voir la
Licence Publique Gnrale GNU pour plus de dtails.

Une copie de la Licence Publique Gnrale GNU devrait tre fournie
avec MinD. Si ce n'est pas le cas, voir <http://www.gnu.org/licenses/>.

*/


#include "bricks.h"

#define NOTHING          0
#define STRING_OPENED    1
#define KEYWORD_FOUND    2
#define STRING_CLOSED    3


/**
 * \fn _TCHAR *RetrieveValueInString(_TCHAR *text, _TCHAR **textNext)
 * \brief retourne la premire chane rencontre dans un texte
 *
 * la fonction parcourt le texte jusqu' rencontrer une chane entre guillemets
 * ou sans guillemets (fin de chane au premier espace ou au premier guillemet rencontr
 * aprs le dbut de la chane)
 *
 * \param text : texte termin par un caractre nul
 * \param textNext : pointeur recevant l'adresse de la partie non lue de la chane
 *
 * \return une chane qui doit tre libre par l'appelant
 * \return une chane statique vide "" si rien n'est trouv
 */
_TCHAR *RetrieveValueInString(_TCHAR *text, _TCHAR **textNext)
{
  _TCHAR *result = NULL;
  if (textNext)
    {
      textNext[0] = text;
    }

  if (text)
    {
      int quote = 0;

      _TCHAR *textPtr = text;

      while (textPtr[0])
        {
          if (!quote)
            {
              if ((textPtr[0] == '\'') || (textPtr[0] == '\"'))
                {
                  if (!result)
                    {
                      quote = textPtr[0];
                      result = &textPtr[1];
                    }
                  else
                    {
                      break;
                    }
                }

              else if (!result && (!_istspace(textPtr[0])))
                {
                  result = textPtr;
                }

              else if (result && _istspace(textPtr[0]))
                {
                  break;
                }
            }

          else if (textPtr[0] == quote)
            {
              if (textNext)
                {
                  textNext[0] = &textPtr[1];
                }

              break;
            }

          ++ textPtr;

          if (textNext)
            {
              textNext[0] = textPtr;
            }
        }

      if (result)
        {
          int textChar = textPtr[0];
          textPtr[0] = '\0';

          if (_tcslen(result))
            {
              result = _tcsdup(result);
            }
          else
            {
              result = NULL;
            }

          textPtr[0] = textChar;
        }
    }

  if (!result)
    {
      result = TEXT("");
    }

  return result;
}


/**
 * \fn _TCHAR *LocateKeywordInString(_TCHAR *text, _TCHAR *keyWord, _TCHAR *stringOpen, _TCHAR *stringClose)
 * \brief localise un mot-clef dans une chane
 *
 * la fonction parcourt la chane jusqu' rencontrer l'ouverture (si dfinie),
 * puis le mot-clef prcd d'ventuels espaces intercalaires,
 * mais la recherche s'arrte si la clture de chane est rencontre.
 * Le mot-clef ne doit pas tre suivi ni prcd
 * de caractres alphanumriques ou d'un underscore.
 * La casse des caractres n'a pas d'importance (majuscule = minuscule).
 *
 * \param text : chane termine par un caractre nul
 * \param keyWord : mot-clef
 * \param stringOpen : mot-clef caractrisant l'ouverture de la chane (peut tre NULL)
 * \param stringClose : mot-clef caractrisant la fermeture de la chane (peut tre NULL)
 *
 * \return pointeur sur le caractre suivant le mot-clef
 * \return NULL si l'item n'est pas trouv
 */
_TCHAR *LocateKeywordInString(_TCHAR *text, _TCHAR *keyWord, _TCHAR *stringOpen, _TCHAR *stringClose)
{
  _TCHAR *result = NULL;

  if (text && keyWord && keyWord[0])
    {
      int state = NOTHING;
      int quote = '\0';
      int textPreviousChar = ' ';

      _TCHAR *textPtr = text;

      while (!result && textPtr[0] && (state != STRING_CLOSED))
        {
          if (!quote)
            {
              if (!stringClose || !stringClose[0] || _tcsnicmp(stringClose, textPtr, _tcslen(stringClose)))
                {
                  if ((textPtr[0] == '\'') || (textPtr[0] == '\"'))
                    {
                      quote = textPtr[0];
                    }

                  else
                    {
                      switch (state)
                        {
                          case NOTHING :
                            {
                              if (stringOpen && stringOpen[0])
                                {
                                  int length = _tcslen(stringOpen);

                                  if (!_tcsnicmp(stringOpen, textPtr, length))
                                    {
                                      textPreviousChar = textPtr[length - 1];
                                      textPtr += length;
                                    }
                                  else
                                    {
                                      break;
                                    }
                                }

                              state = STRING_OPENED;
                            }

                          case STRING_OPENED :
                            {
                              if (!_istalnum(textPreviousChar) && (textPreviousChar != '_'))
                                {
                                  if (!_tcsnicmp(keyWord, textPtr, _tcslen(keyWord)))
                                    {
                                      state = KEYWORD_FOUND;
                                      textPtr += _tcslen(keyWord);
                                    }

                                  else
                                    {
                                      if (stringClose && stringClose[0])
                                        {
                                          if (!_tcsnicmp(stringClose, textPtr, _tcslen(stringClose)))
                                            {
                                              state = STRING_CLOSED;
                                            }
                                        }

                                      else if (!_istspace(textPtr[0]))
                                        {
                                          state = NOTHING;
                                        }

                                      break;
                                    }
                                }

                              else
                                {
                                  state = NOTHING;
                                  break;
                                }
                            }

                          case KEYWORD_FOUND :
                            {
                              if (!_istalnum(textPtr[0]) && (textPtr[0] != '_'))
                                {
                                  result = textPtr;
                                }
                              else
                                {
                                  state = NOTHING;
                                }

                              break;
                            }

                          default :
                            {
                              break;
                            }
                        }
                    }
                }

              else if (stringClose && !_tcsnicmp(textPtr, stringClose, _tcslen(stringClose)))
                {
                  state = STRING_CLOSED;
                }
            }

          else if (textPtr[0] == quote)
            {
              quote = 0;
            }

          textPreviousChar = textPtr[0];
          ++ textPtr;
        }
    }

  return result;
}


/**
 * \fn _TCHAR *FileInChooser(HWND hwnd, const _TCHAR *title, const _TCHAR *type)
 * \brief Fonction permettant de slectionner un fichier
 *
 * \param title : titre de la boite de dialogue
 * \param hwnd : fentre parent
 * \param type : type du fichier (par exemple "OpenDocument Texte (*.odt)\0*.odt\0\0")
 *
* \return le chemin du fichier s'il a t slectionn (mmoire  librer par l'appelant)
 * \return NULL dans le cas contraire
 */
_TCHAR *FileInChooser(HWND hwnd, const _TCHAR *title, const _TCHAR *type)
{
  _TCHAR *fileName = calloc(MAX_PATH, sizeof(_TCHAR));

  if (fileName != NULL)
    {
      if (!title)
        {
          title = TEXT("Ouvrir un fichier ...");
        }

#ifndef MIND_GTK
      OPENFILENAME ofn;

      ZeroMemory(&ofn, sizeof(ofn));

      ofn.lStructSize = sizeof(ofn);
      ofn.hwndOwner = hwnd;
      ofn.lpstrFilter = type;
      ofn.lpstrFile = fileName;
      ofn.nMaxFile = MAX_PATH;
      ofn.lpstrDefExt = TEXT("");
      ofn.lpstrTitle = title;

      ofn.Flags = OFN_EXPLORER
                  | OFN_FILEMUSTEXIST
                  | OFN_HIDEREADONLY;

      if (GetOpenFileName(&ofn))
        {
          return fileName;
        }
#else
      GtkWidget *dialog;
      GtkFileChooserAction action = GTK_FILE_CHOOSER_ACTION_OPEN;

      dialog = gtk_file_chooser_dialog_new(title,
                                           GTK_WINDOW(hwnd),
                                           action,
                                           TEXT("Annuler"),
                                           GTK_RESPONSE_CANCEL,
                                           TEXT("Ouvrir"),
                                           GTK_RESPONSE_ACCEPT,
                                           NULL);

      if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT)
        {
          char *gfileName;
          gfileName = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));

          _tcsncpy(fileName, gfileName, MAX_PATH);

          g_free(gfileName);
          gtk_widget_destroy(dialog);

          return fileName;
        }

      gtk_widget_destroy(dialog);
#endif

      free(fileName);
    }

  return NULL;
}


/**
 * \fn _TCHAR *FileOutChooser(HWND hwnd, const _TCHAR *title, const _TCHAR *type)
 * \brief permet de slectionner un chemin pour sauvegarder un fichier
 *
 * Cette fonction permet  l'utilisateur de choisir l'emplacement et le nom d'un fichier  sauvegarder,
 * et autorise ventuellement l'crasement d'un fichier existant.
 *
 * \param title : titre de la boite de dialogue
 * \param hwnd : fentre parent
 * \param type : type du fichier (par exemple "OpenDocument Texte (*.odt)\0*.odt\0\0")
 *
 * \return le chemin du fichier de sauvegarde s'il a t slectionn
 * \return NULL dans le cas contraire
 */
_TCHAR *FileOutChooser(HWND hwnd, const _TCHAR *title, const _TCHAR *type)
{
  _TCHAR *fileName = calloc(MAX_PATH + 1, sizeof(_TCHAR));

  if (fileName != NULL)
    {
      if (!title)
        {
          title = TEXT("Enregistrer sous ...");
        }

#ifndef MIND_GTK
      OPENFILENAME ofn;

      ZeroMemory(&ofn, sizeof(ofn));

      ofn.lStructSize = sizeof(ofn);
      ofn.hwndOwner = hwnd;
      ofn.lpstrFilter = type;
      ofn.lpstrFile = fileName;
      ofn.nMaxFile = MAX_PATH;
      ofn.lpstrDefExt = TEXT("");
      ofn.lpstrTitle = title;

      ofn.Flags = OFN_PATHMUSTEXIST
                  | OFN_HIDEREADONLY
                  | OFN_EXPLORER
                  | OFN_CREATEPROMPT
                  | OFN_NONETWORKBUTTON
                  | OFN_NOREADONLYRETURN;

      int i = 0;
      while ((i = GetSaveFileName(&ofn)))
        {
          if (_taccess(fileName, R_OK) == 0)
            {
              MessageBox(hwnd, TEXT("\n Il existe un fichier portant ce nom ! \n"), TEXT("Erreur !"), MB_ICONINFORMATION);
              continue;
            }

          break;
        }

      if (i != 0)
        {
          return fileName;
        }
#else
      GtkWidget *dialog;
      GtkFileChooserAction action = GTK_FILE_CHOOSER_ACTION_SAVE;

      dialog = gtk_file_chooser_dialog_new(title,
                                           GTK_WINDOW(hwnd),
                                           action,
                                           TEXT("Annuler"),
                                           GTK_RESPONSE_CANCEL,
                                           TEXT("Choisir"),
                                           GTK_RESPONSE_ACCEPT,
                                           NULL);

      gtk_file_chooser_set_do_overwrite_confirmation(GTK_FILE_CHOOSER(dialog), TRUE);

      if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT)
        {
          char *gfileName;
          gfileName = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));

          _tcsncpy(fileName, gfileName, MAX_PATH);

          g_free(gfileName);
          gtk_widget_destroy(dialog);

          return fileName;
        }

      gtk_widget_destroy(dialog);
#endif

      free(fileName);
    }

  return NULL;
}


/**
 * \fn HWND AddTooltip(HWND hwnd, const _TCHAR *text)
 * \brief ajoute une bulle d'aide  une fentre
 *
 * \param hwnd : handle de la fentre
 * \param text : texte de la bulle
 *
 * \return le handle de la bulle
 */
HWND AddTooltip(HWND hwnd, const _TCHAR *text)
{
  HWND hwndTip = NULL;

  if (hwnd && text)
    {
      // Cre la bulle

#ifndef MIND_GTK
      hwndTip = CreateWindowEx(0, TOOLTIPS_CLASS, NULL,
                               WS_POPUP | TTS_ALWAYSTIP | TTS_BALLOON,
                               CW_USEDEFAULT, CW_USEDEFAULT,
                               CW_USEDEFAULT, CW_USEDEFAULT,
                               (HWND)hwnd, NULL,
                               (HINSTANCE)GetWindowLongPtr(hwnd, GWLP_HINSTANCE), NULL);

      if (!hwnd || !hwndTip)
        {
          return (HWND)NULL;
        }

      // Associe le tooltip avec la fentre

      TOOLINFO toolInfo = { 0 };
      toolInfo.cbSize = sizeof(toolInfo);
      toolInfo.hwnd = hwnd;
      toolInfo.uFlags = TTF_IDISHWND | TTF_SUBCLASS;
      toolInfo.uId = (UINT_PTR)hwnd;
      toolInfo.lpszText = (_TCHAR *) text;
      SendMessage(hwndTip, TTM_ADDTOOL, 0, (LPARAM)&toolInfo);
#else
#endif
    }

  return hwndTip;
}


/**
 * \fn void MessageBoxYesNo(HWND hwndParent, _TCHAR *message, _TCHAR *title)
 * \brief remplace sous GTK la fonction Windows MessageBox avec YESNO
 *
 * \param hwndParent : handle de la fentre parent
 * \param message : message  afficher
 * \param title : titre du message
 *
 * \return le choix de l'utilisateur
 */
int MessageBoxYesNo(HWND hwndParent, _TCHAR *message, _TCHAR *title)
{
  int result = 0;

  if (!title)
    {
      title = TEXT("");
    }

#ifndef MIND_GTK

  result = MessageBox(hwndParent, message, title, MB_TOPMOST | MB_ICONQUESTION | MB_YESNO);

#else

  GtkWidget *hwnd = gtk_dialog_new_with_buttons (title,
                                                 GTK_WINDOW(hwndParent),
                                                 GTK_DIALOG_MODAL,
                                                 "Oui", IDYES,
                                                 "Non", IDNO,
                                                 NULL);

  gtk_window_set_position(GTK_WINDOW(hwnd), GTK_WIN_POS_CENTER);

  GtkWidget *client = gtk_dialog_get_content_area(GTK_DIALOG(hwnd));
  gtk_container_set_border_width(GTK_CONTAINER(client), 8);

  GtkWidget *label = gtk_label_new(message);
  gtk_container_add(GTK_CONTAINER(client), label);

  gtk_widget_show_all(hwnd);
  result = gtk_dialog_run(GTK_DIALOG(hwnd));
  gtk_widget_destroy(GTK_WIDGET(hwnd));
#endif

  return result;
}


