/**
 * \file dialog.c
 *
 * \brief Implmentation des fonctions et macros pour la gestion des dialogues
 * \author Denis MARTIN
 * \date 2017-2019
 */


/*

Copyright (C) 2017, 2019 Denis MARTIN

---------------------------------------

This file is part of Mind.

Mind is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or any later version.

Mind is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Mind.  If not, see <http://www.gnu.org/licenses/>.

--------------------------------------------------------------------------

Ce fichier fait partie de Mind.

Mind est un logiciel libre : il peut tre redistribu et/ou modifi
selon les termes de la Licence Publique Gnrale GNU telle que publie par
la Free Software Foundation, soit la version 3 de la licence ou toute
version ultrieure.

Mind est distribu dans l'espoir qu'il sera utile,
mais SANS AUCUNE GARANTIE ; sans mme la garantie implicite de
VALEUR MARCHANDE ou d'ADEQUATION AVEC UN BUT PARTICULIER. Voir la
Licence Publique Gnrale GNU pour plus de dtails.

Une copie de la Licence Publique Gnrale GNU devrait tre fournie
avec Mind. Si ce n'est pas le cas, voir <http://www.gnu.org/licenses/>.

*/


#include "dialog.h"


#ifndef MIND_GTK
#else
static void MinDDialog_OnCommand(HWND hwndCtl, DATA *data);
#endif


/**
 * \fn _TCHAR *MinDDialog_ExtractVariable(_TCHAR *text)
 * \brief extrait le nom d'une variable d'une commande de dialogue
 *
 * la fonction parcourt la commande jusqu' rencontrer le mot-clef "TO"
 * puis extrait le nom qui suit (dans ou hors guillemets).
 * La recherche s'arrte si une parenthse fermante est atteinte.
 *
 * \param text : texte de la commande
 *
 * \return une chane qui doit tre libre par l'appelant
 * \return une chane statique vide "" si le mot-clef est trouv sans argument
 * \return NULL si le mot-clef n'est pas trouv (ou autre problme)
 */
static _TCHAR *MinDDialog_ExtractVariable(_TCHAR *text)
{
  _TCHAR *result = LocateKeywordInString(text, TEXT("TO"), NULL, TEXT(")"));

  if (result)
    {
      result = RetrieveValueInString(result, NULL);
    }

  return result;
}


/**
 * \fn POINT MinDDialog_ExtractPoint(_TCHAR *text, _TCHAR *keyWord)
 * \brief extrait des coordonnes d'un script de dialogue
 *
 * la fonction parcourt la commande jusqu' rencontrer le mot-clef puis extrait les deux entiers situs derrire, mais
 * la fonction s'interrompt et retourne zro si une parenthse fermante est rencontre avant le mot-clef, par ailleurs si
 * un seul entier est trouv, le second a pour valeur zro
 *
 * \param text : texte de la commande
 * \param keyWord : mot-clef  trouver
 *
 * \return coordonnes ou (0, 0) en cas de problme
 */
static POINT MinDDialog_ExtractPoint(_TCHAR *text, _TCHAR *keyWord)
{
  POINT result;
  result.x = 0;
  result.y = 0;

  _TCHAR *textPtr = LocateKeywordInString(text, keyWord, NULL, TEXT(")"));

  if (textPtr)
    {
      _TCHAR *textPtrY = NULL;
      result.x = _tcstol(textPtr, &textPtrY, 0);
      result.y = _tcstol(textPtrY, NULL, 0);
    }

  return result;
}


/**
 * \fn long MinDDialog_ExtractInt(_TCHAR *text, _TCHAR *keyWord)
 * \brief extrait un entier d'un script de dialogue
 *
 * la fonction parcourt la commande jusqu' rencontrer le mot-clef puis extrait l'entier situ derrire, mais
 * la fonction s'interrompt et retourne zro si une parenthse fermante est rencontre avant le mot-clef
 *
 * \param text : texte de la commande
 * \param keyWord : mot-clef  trouver
 *
 * \return valeur entire ou 0 en cas de problme
 */
static long MinDDialog_ExtractInt(_TCHAR *text, _TCHAR *keyWord)
{
  long result = 0;

  _TCHAR *textPtr = LocateKeywordInString(text, keyWord, NULL, TEXT(")"));

  if (textPtr)
    {
      result = _tcstol(textPtr, NULL, 0);
    }

  return result;
}


/**
 * \fn _TCHAR *MinDDialog_ExtractString(_TCHAR *text, _TCHAR *keyWord)
 * \brief extrait une chane d'un script de dialogue
 *
 * la fonction parcourt la commande jusqu' rencontrer le mot-clef puis extrait la chane situe derrire, mais
 * la fonction s'interrompt et retourne NULL si une parenthse fermante est rencontre avant le mot-clef
 *
 * \param text : texte de la commande
 * \param keyWord : mot-clef  trouver
 *
 * \return une chane qui doit tre libre par l'appelant
 * \return une chane statique vide "" si le mot-clef est trouv sans argument
 * \return NULL si le mot-clef n'est pas trouv (ou autre problme)
 */
static _TCHAR *MinDDialog_ExtractString(_TCHAR *text, _TCHAR *keyWord)
{
  _TCHAR *result = LocateKeywordInString(text, keyWord, NULL, TEXT(")"));

  if (result)
    {
      result = RetrieveValueInString(result, NULL);
    }

  return result;
}


/**
 * \fn _TCHAR *MinDDialog_LocateItem(_TCHAR *text, _TCHAR *keyWord)
 * \brief localise un item dans un script de dialogue
 *
 * la fonction parcourt la commande jusqu' rencontrer le mot-clef,
 * prcd d'une parenthse et d'ventuels espaces intercalaires
 *
 * \param text : texte de la commande
 * \param keyWord : mot-clef  trouver
 *
 * \return pointeur sur le caractre suivant le mot-clef
 * \return NULL si l'item n'est pas trouv
 */
static _TCHAR *MinDDialog_LocateItem(_TCHAR *text, _TCHAR *keyWord)
{
  _TCHAR *result = LocateKeywordInString(text, keyWord, TEXT("("), NULL);

  return result;
}


/**
 * \fn BOOL MinDDialog_CreateStatic(HWND hwnd, _TCHAR *text, LONG_PTR itemId)
 * \brief Cration d'un contrle statique (texte)
 *
 * \param hwnd : handle de la boite de dialogue
 * \param text : texte de la commande
 * \param itemId : numro d'item
 *
 * \return TRUE si tout s'est bien pass
 * \return FALSE dans le cas contraire
 */
static BOOL MinDDialog_CreateStatic(HWND hwnd, _TCHAR *text, LONG_PTR itemId)
{
  BOOL result = FALSE;

  POINT pos = MinDDialog_ExtractPoint(text, TEXT("AT"));

  _TCHAR *title = MinDDialog_ExtractString(text, TEXT("TITLE"));
  if (title && title[0])
    {
      int width = MinDDialog_ExtractInt(text, TEXT("WIDTH"));
      int height = MinDDialog_ExtractInt(text, TEXT("HEIGHT"));

#ifndef MIND_GTK
      int defaultWidth = 5, defaultHeight = 20;

      DWORD style = WS_CHILD | WS_VISIBLE | WS_TABSTOP;
#else
#endif

      _TCHAR * center = MinDDialog_ExtractString(text, TEXT("CENTER"));
      if (center)
        {
#ifndef MIND_GTK
          style |=  SS_CENTER;
#else
#endif
          if (center[0])
            {
              free(center);
            }
        }

      _TCHAR * right = MinDDialog_ExtractString(text, TEXT("RIGHT"));
      if (right)
        {
#ifndef MIND_GTK
          style |=  SS_RIGHT;
#else
#endif
          if (right[0])
            {
              free(right);
            }
        }

      HWND hwndItem = NULL;

#ifndef MIND_GTK
      hwndItem = CreateWindowEx(0, TEXT("STATIC"), title, style,
                                pos.x, pos.y, defaultWidth, defaultHeight,
                                hwnd, (HMENU) itemId, (HINSTANCE)GetWindowLongPtr(hwnd, GWLP_HINSTANCE), NULL);
#else
      hwndItem = gtk_label_new(title);

      gtk_fixed_put(GTK_FIXED(clientZone), hwndItem, pos.x, pos.y);

      if (width && height)
        {
          gtk_label_set_line_wrap(GTK_LABEL(hwndItem), TRUE);
          gtk_widget_set_size_request(hwndItem, width, height);
        }
#endif

      if (hwndItem)
        {
          result = TRUE;

#ifndef MIND_GTK
          PostMessage(hwndItem, WM_SETFONT, (WPARAM)GetStockObject(DEFAULT_GUI_FONT), MAKELPARAM(TRUE, 0));

          // adapte la taille de la fentre

          RECT rClient;
          GetClientRect(hwndItem, &rClient);

          int textSizeX = rClient.right;
          int textSizeY = rClient.bottom;

          SIZE textSize;
          HDC hdc = GetDC(hwndItem);
          HFONT hfnt = SelectFont(hdc, GetStockObject(DEFAULT_GUI_FONT));

          GetTextExtentPoint32(hdc, title, _tcslen(title), &textSize);

          hfnt = SelectFont(hdc, hfnt);
          DeleteObject(hfnt);

          textSizeX = textSize.cx;
          textSizeY = textSize.cy;

          if (!width)
            {
              int idealWidth = textSizeX + (defaultWidth - rClient.right);
              width = (idealWidth > defaultWidth) ? idealWidth : defaultWidth;
            }

          if (!height)
            {
              int idealHeight = textSizeY + (defaultHeight - rClient.bottom);
              height = (idealHeight > defaultHeight) ? idealHeight : defaultHeight;
            }

          MoveWindow(hwndItem, pos.x, pos.y, width, height, TRUE);
#else
#endif
        }

      free(title);
    }

  return result;
}


/**
 * \fn BOOL MinDDialog_CreateEditBox(HWND hwnd, _TCHAR *text, LONG_PTR itemId)
 * \brief Cration d'un contrle d'dition
 *
 * \param hwnd : handle de la boite de dialogue
 * \param text : texte de la commande
 * \param itemId : numro d'item
 *
 * \return TRUE si tout s'est bien pass
 * \return FALSE dans le cas contraire
 */
static BOOL MinDDialog_CreateEditBox(HWND hwnd, _TCHAR *text, LONG_PTR itemId)
{
  BOOL result = FALSE;

  POINT pos = MinDDialog_ExtractPoint(text, TEXT("AT"));

  int width = MinDDialog_ExtractInt(text, TEXT("WIDTH"));
  int height = MinDDialog_ExtractInt(text, TEXT("HEIGHT"));

  _TCHAR *title = MinDDialog_ExtractString(text, TEXT("DEFAULT"));
  if (!title)
    {
      title = TEXT("");
    }

  // initialise la variable correspondante

  _TCHAR *variable = MinDDialog_ExtractVariable(text);
  if (variable && variable[0])
    {
      MinDVar *mindVar = MinDVariable_Retrieve(&mindVarGlobal, variable, 0);
      if (mindVar && mindVar->value)
        {
          if (title[0])
            {
              free(title);
              title = NULL;
            }

          if (mindVar->value[0])
            {
              title = _tcsdup(mindVar->value);
            }

          if (!title)
            {
              title = TEXT("");
            }
        }

      MinDVariable_Fill(&mindVarGlobal, 'C', variable, itemId, title);
      free(variable);
    }

  HWND hwndItem = NULL;
#ifndef MIND_GTK
  int defaultWidth = 120, defaultHeight = 24;

  hwndItem = CreateWindowEx(WS_EX_STATICEDGE | WS_EX_CLIENTEDGE,
                            TEXT("EDIT"), title, WS_CHILD | WS_VISIBLE | WS_TABSTOP | ES_AUTOHSCROLL,
                            pos.x, pos.y, defaultWidth, defaultHeight,
                            hwnd, (HMENU) itemId, (HINSTANCE)GetWindowLongPtr(hwnd, GWLP_HINSTANCE), NULL);
#else
  hwndItem = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(hwndItem), title);

  if (width && height)
    {
      gtk_widget_set_size_request(hwndItem, width, height);
    }

  gtk_fixed_put(GTK_FIXED(clientZone), hwndItem, pos.x, pos.y);

  static DATA data[MAX_ITEM];
  int itemNum = ITEM_INDEX_MASK & itemId;
  if (itemNum <= MAX_ITEM)
    {
      data[itemNum - 1].id = itemId;
      data[itemNum - 1].hwnd = hwnd;

      g_signal_connect(hwndItem, "changed", G_CALLBACK(MinDDialog_OnCommand), &data[itemNum - 1]);
    }
#endif

  if (hwndItem)
    {
      result = TRUE;
#ifndef MIND_GTK
      PostMessage(hwndItem, WM_SETFONT, (WPARAM)GetStockObject(DEFAULT_GUI_FONT), MAKELPARAM(TRUE, 0));


      // adapte la taille de la fentre

      RECT rClient;
      GetClientRect(hwndItem, &rClient);

      int textSizeX = rClient.right;
      int textSizeY = rClient.bottom;

      SIZE textSize;
      HDC hdc = GetDC(hwndItem);
      HFONT hfnt = SelectFont(hdc, GetStockObject(DEFAULT_GUI_FONT));

      GetTextExtentPoint32(hdc, title, _tcslen(title), &textSize);

      hfnt = SelectFont(hdc, hfnt);
      DeleteObject(hfnt);

      textSizeX = textSize.cx;
      textSizeY = textSize.cy;

      if (!width)
        {
          int idealWidth = textSizeX + (defaultWidth - rClient.right) + 2;
          width = (idealWidth > defaultWidth) ? idealWidth : defaultWidth;
        }

      if (!height)
        {
          int idealHeight = textSizeY + (defaultHeight - rClient.bottom);
          height = (idealHeight > defaultHeight) ? idealHeight : defaultHeight;
        }

      MoveWindow(hwndItem, pos.x, pos.y, width, height, TRUE);
#else
#endif
    }

  if (title[0])
    {
      free(title);
    }

  return result;
}


/**
 * \fn BOOL MinDDialog_CreateCheckBox(HWND hwnd, _TCHAR *text, LONG_PTR itemId)
 * \brief Cration d'une case  cocher
 *
 * \param hwnd : handle de la boite de dialogue
 * \param text : texte de la commande
 * \param itemId : numro d'item
 *
 * \return TRUE si tout s'est bien pass
 * \return FALSE dans le cas contraire
 */
static BOOL MinDDialog_CreateCheckBox(HWND hwnd, _TCHAR *text, LONG_PTR itemId)
{
  BOOL result = FALSE;

  POINT pos = MinDDialog_ExtractPoint(text, TEXT("AT"));

  int width = MinDDialog_ExtractInt(text, TEXT("WIDTH"));
  int height = MinDDialog_ExtractInt(text, TEXT("HEIGHT"));

  _TCHAR *title = MinDDialog_ExtractString(text, TEXT("TITLE"));
  if (!title)
    {
      title = TEXT("");
    }

  _TCHAR *check = MinDDialog_ExtractString(text, TEXT("CHECKED"));
  if (check && check[0])
    {
      free(check);
      check = TEXT("");
    }

  // gestion de la variable lie

  _TCHAR *variable = MinDDialog_ExtractVariable(text);
  if (variable && variable[0])
    {
      MinDVar *mindVar = MinDVariable_Retrieve(&mindVarGlobal, variable, 0);

      if (mindVar)
        {
          if ((mindVar->value) && (mindVar->value[0] != 'F'))
            {
              check = TEXT("");
            }
          else
            {
              check = NULL;
            }
        }

      MinDVariable_Fill(&mindVarGlobal, 'L', variable, itemId, check ? TEXT("TRUE") : TEXT("FALSE"));

      free(variable);
    }

  // cration de la fentre
  HWND hwndItem = NULL;

#ifndef MIND_GTK
  int defaultWidth = 120, defaultHeight = 24;

  hwndItem = CreateWindowEx(0, TEXT("BUTTON"), title, WS_CHILD | WS_VISIBLE | WS_TABSTOP | BS_AUTOCHECKBOX,
                            pos.x, pos.y, defaultWidth, defaultHeight,
                            hwnd, (HMENU) itemId, (HINSTANCE)GetWindowLongPtr(hwnd, GWLP_HINSTANCE), NULL);
#else
  hwndItem = gtk_check_button_new_with_label(title);
  gtk_fixed_put(GTK_FIXED(clientZone), hwndItem, pos.x, pos.y);
  if (width && height)
    {
      gtk_widget_set_size_request(hwndItem, width, height);
    }

  static DATA data[MAX_ITEM];
  int itemNum = ITEM_INDEX_MASK & itemId;
  if (itemNum <= MAX_ITEM)
    {
      data[itemNum - 1].id = itemId;
      data[itemNum - 1].hwnd = hwnd;

      g_signal_connect(hwndItem, "clicked", G_CALLBACK(MinDDialog_OnCommand), &data[itemNum - 1]);
    }
#endif

  if (hwndItem)
    {
      result = TRUE;

      if (check)
        {
#ifndef MIND_GTK
          PostMessage(hwndItem, BM_SETCHECK, (WPARAM) BST_CHECKED, MAKELPARAM(TRUE, 0));
#else
          gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(hwndItem), TRUE);
#endif
        }

#ifndef MIND_GTK
      PostMessage(hwndItem, WM_SETFONT, (WPARAM)GetStockObject(DEFAULT_GUI_FONT), MAKELPARAM(TRUE, 0));

      // adapte la taille de la fentre

      RECT rClient;
      GetClientRect(hwndItem, &rClient);

      int textSizeX = rClient.right;
      int textSizeY = rClient.bottom;

      int checkSizeX = defaultWidth, checkSizeY = defaultHeight;

      SIZE textSize;
      HDC hdc = GetDC(hwndItem);
      HFONT hfnt = SelectFont(hdc, GetStockObject(DEFAULT_GUI_FONT));

      GetTextExtentPoint32(hdc, title, _tcslen(title), &textSize);

      hfnt = SelectFont(hdc, hfnt);
      DeleteObject(hfnt);

      textSizeX = textSize.cx;
      textSizeY = textSize.cy;

      checkSizeX = GetSystemMetrics(SM_CXMENUCHECK);
      checkSizeY = GetSystemMetrics(SM_CYMENUCHECK);

      if (!width)
        {
          int idealWidth = textSizeX + (defaultWidth - rClient.right) + checkSizeX + 2;
          width = (idealWidth > defaultWidth) ? idealWidth : defaultWidth;
        }

      if (!height)
        {
          int idealHeight = textSizeY + (defaultHeight - rClient.bottom);
          idealHeight = (checkSizeY > idealHeight) ? checkSizeY : idealHeight;
          height = (idealHeight > defaultHeight) ? idealHeight : defaultHeight;
        }

      MoveWindow(hwndItem, pos.x, pos.y, width, height, TRUE);
#else
#endif
    }

  if (title[0])
    {
      free(title);
    }

  return result;
}


/**
 * \fn BOOL MinDDialog_CreateDropBox(HWND hwnd, _TCHAR *text, LONG_PTR itemId)
 * \brief Cration d'une liste droulante
 *
 * \param hwnd : handle de la boite de dialogue
 * \param text : texte de la commande
 * \param itemId : numro d'item
 *
 * \return TRUE si tout s'est bien pass
 * \return FALSE dans le cas contraire
 */
static BOOL MinDDialog_CreateDropBox(HWND hwnd, _TCHAR *text, LONG_PTR itemId)
{
  BOOL result = FALSE;

  POINT pos = MinDDialog_ExtractPoint(text, TEXT("AT"));

  int width = MinDDialog_ExtractInt(text, TEXT("WIDTH"));
  int height = MinDDialog_ExtractInt(text, TEXT("HEIGHT"));

  // initialise la variable correspondante

  _TCHAR *defaultString = NULL;

  _TCHAR *variable = MinDDialog_ExtractVariable(text);
  if (variable && variable[0])
    {
      MinDVar *mindVar = MinDVariable_Retrieve(&mindVarGlobal, variable, 0);
      if (mindVar && mindVar->value)
        {
          defaultString = _tcsdup(mindVar->value);
        }

      MinDVariable_Fill(&mindVarGlobal, 'C', variable, itemId, defaultString);
      free(variable);
    }


  _TCHAR *title = MinDDialog_ExtractString(text, TEXT("TITLE"));
  if (title && title[0])
    {
      HWND hwndItem = NULL;
      _TCHAR *titlePtr = title - 1;

#ifndef MIND_GTK
      int defaultWidth = 120, defaultHeight = 100;
      int textSizeMaxW = 0, textSizeMaxH = 0;
#else
#endif

      do
        {
          ++ titlePtr;
          int length = _tcscspn(titlePtr, sSeparator);
          if (length)
            {
#ifndef MIND_GTK
              SIZE textSize;
              ZeroMemory(&textSize, sizeof(textSize));
              HDC hdc = GetDC(hwnd);
              HFONT hfnt = SelectFont(hdc, GetStockObject(DEFAULT_GUI_FONT));

              GetTextExtentPoint32(hdc, titlePtr, length, &textSize);
              textSizeMaxW = (textSize.cx > textSizeMaxW) ? textSize.cx : textSizeMaxW;
              textSizeMaxH = (textSize.cy > textSizeMaxH) ? textSize.cy : textSizeMaxH;

              hfnt = SelectFont(hdc, hfnt);
              DeleteObject(hfnt);
#else
#endif
            }
        }
      while ((titlePtr = _tcschr(titlePtr, cSeparator)));

#ifndef MIND_GTK
      hwndItem = CreateWindowEx(0, TEXT("COMBOBOX"), TEXT(""), WS_CHILD | WS_VISIBLE | WS_TABSTOP | CBS_DROPDOWNLIST | CBS_HASSTRINGS,
                                pos.x, pos.y, defaultWidth, defaultHeight,
                                hwnd, (HMENU) itemId, (HINSTANCE)GetWindowLongPtr(hwnd, GWLP_HINSTANCE), NULL);
#else
      hwndItem = gtk_combo_box_text_new();

      if (width && height)
        {
#if GTK_MAJOR_VERSION > 2
          gtk_widget_set_size_request(hwndItem, width, height);
#else
          // il n'est pas souhaitable de fixer les dimensions avec GTK2
#endif
        }

      gtk_fixed_put(GTK_FIXED(clientZone), hwndItem, pos.x, pos.y);

      static DATA data[MAX_ITEM];
      int itemGtkNum = ITEM_INDEX_MASK & itemId;
      if (itemGtkNum <= MAX_ITEM)
        {
          data[itemGtkNum - 1].id = itemId;
          data[itemGtkNum - 1].hwnd = hwnd;

          g_signal_connect(hwndItem, "changed", G_CALLBACK(MinDDialog_OnCommand), &data[itemGtkNum - 1]);
        }
#endif

      if (hwndItem)
        {
          result = TRUE;
#ifndef MIND_GTK
          PostMessage(hwndItem, WM_SETFONT, (WPARAM)GetStockObject(DEFAULT_GUI_FONT), MAKELPARAM(TRUE, 0));
#else
#endif

          titlePtr = title - 1;

          int defaultItem = MinDDialog_ExtractInt(text, TEXT("DEFAULT")) - 1;
          int itemNum = 0;

          do
            {
              ++ titlePtr;
              int length = _tcscspn(titlePtr, sSeparator);
              _TCHAR titleTmp[length + 1];
              _tcsncpy(titleTmp, titlePtr, length);
              titleTmp[length] = '\0';

#ifndef MIND_GTK
              SendMessage(hwndItem, CB_ADDSTRING, (WPARAM) 0, (LPARAM) titleTmp);
#else
              gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(hwndItem), titleTmp);
#endif

              if (defaultString)
                {
                  if (!_tcscmp(defaultString, titleTmp))
                    {
                      defaultItem = itemNum;
                      free(defaultString);
                      defaultString = NULL;
                    }
                }
              else if (itemNum == defaultItem)
                {
                  MinDVariable_Fill(&mindVarGlobal, 'C', NULL, itemId, titleTmp);
                }

              ++ itemNum;
            }
          while ((titlePtr = _tcschr(titlePtr, cSeparator)));

#ifndef MIND_GTK
          PostMessage(hwndItem, CB_SETCURSEL, (WPARAM)defaultItem, (LPARAM)0);
#else
          gtk_combo_box_set_active(GTK_COMBO_BOX(hwndItem), defaultItem);
#endif

#ifndef MIND_GTK
          // adapte la taille de la fentre

          RECT rClient;
          GetClientRect(hwndItem, &rClient);

          if (!width)
            {
              int idealWidth = textSizeMaxW + (defaultWidth - rClient.right) + 24;
              width = (idealWidth > defaultWidth) ? idealWidth : defaultWidth;
            }

          if (!height)
            {
              int idealHeight = textSizeMaxH + (defaultHeight - rClient.bottom);
              height = (idealHeight > defaultHeight) ? idealHeight : defaultHeight;
            }

          MoveWindow(hwndItem, pos.x, pos.y, width, height, TRUE);
#else
#endif
        }

      free(title);
    }

  return result;
}


/**
 * \fn int MinDDialog_CreateButtonSet(HWND hwnd, _TCHAR *text, LONG_PTR itemId)
 * \brief Cration d'un bloc de boutons
 *
 * \param hwnd : handle de la boite de dialogue
 * \param text : texte de la commande
 * \param itemId : numro d'item
 *
 * \return le nombre de boutons crs
 */
static int MinDDialog_CreateButtonSet(HWND hwnd, _TCHAR *text, LONG_PTR itemId)
{
  int result = 0;

  POINT pos = MinDDialog_ExtractPoint(text, TEXT("AT"));
  _TCHAR * horizontal = MinDDialog_ExtractString(text, TEXT("HORZ"));
  int defaultButton = MinDDialog_ExtractInt(text, TEXT("DEFAULT"));

  _TCHAR *title = MinDDialog_ExtractString(text, TEXT("TITLE"));
  if (!title)
    {
      title = _tcsdup(TEXT("&OK;&Abandon"));
    }
  else if (!title[0])
    {
      title = _tcsdup(TEXT("&OK"));
    }

  if (title)
    {
#ifndef MIND_GTK
      int defaultWidth = 80, defaultHeight = 24;
      int textSizeMaxW = 0, textSizeMaxH = 0;
#else
#endif

      _TCHAR *titlePtr = title - 1;
      int itemNb = 0;

      do
        {
          ++ titlePtr;
          int length = _tcscspn(titlePtr, sSeparator);
          if (length)
            {
#ifndef MIND_GTK
              SIZE textSize;
              ZeroMemory(&textSize, sizeof(textSize));
              HDC hdc = GetDC(hwnd);
              HFONT hfnt = SelectFont(hdc, GetStockObject(DEFAULT_GUI_FONT));

              GetTextExtentPoint32(hdc, titlePtr, length, &textSize);
              textSizeMaxW = (textSize.cx > textSizeMaxW) ? textSize.cx : textSizeMaxW;
              textSizeMaxH = (textSize.cy > textSizeMaxH) ? textSize.cy : textSizeMaxH;

              hfnt = SelectFont(hdc, hfnt);
              DeleteObject(hfnt);
#else
#endif
            }

          ++ itemNb;
        }
      while ((titlePtr = _tcschr(titlePtr, cSeparator)));

      int width = MinDDialog_ExtractInt(text, TEXT("WIDTH"));
      int height = MinDDialog_ExtractInt(text, TEXT("HEIGHT"));

#ifndef MIND_GTK
      width /= (horizontal ? itemNb : 1);
      height /= (horizontal ? 1 : itemNb);
#else
      GtkWidget * box = NULL;
#if GTK_MAJOR_VERSION > 2
      box = gtk_box_new(horizontal ? GTK_ORIENTATION_HORIZONTAL : GTK_ORIENTATION_VERTICAL, 2);
      gtk_box_set_homogeneous(GTK_BOX(box), TRUE);
#else
      box = horizontal ? gtk_hbox_new(TRUE, 2) : gtk_vbox_new(TRUE, 2);
#endif
      if (width && height)
        {
          gtk_widget_set_size_request(box, width, height);
        }

      gtk_fixed_put(GTK_FIXED(clientZone), box, pos.x, pos.y);
#endif

      // boucle de cration des boutons

      titlePtr = title - 1;
      int itemNum = 0;

      do
        {
          ++ titlePtr;
          int length = _tcscspn(titlePtr, sSeparator);
          _TCHAR titleTmp[length + 1];
          _tcsncpy(titleTmp, titlePtr, length);
          titleTmp[length] = '\0';

          ++ itemId;
          ++ itemNum;

#ifndef MIND_GTK
          DWORD style = WS_CHILD | WS_VISIBLE | WS_TABSTOP;
#else
#endif
          if (defaultButton == itemNum)
            {
#ifndef MIND_GTK
              style |= BS_DEFPUSHBUTTON;
#else
#endif
            }
          HWND hwndItem = NULL;

#ifndef MIND_GTK
          hwndItem = CreateWindowEx(0, TEXT("BUTTON"), titleTmp, style,
                                    pos.x, pos.y, defaultWidth, defaultHeight,
                                    hwnd, (HMENU) itemId, (HINSTANCE)GetWindowLongPtr(hwnd, GWLP_HINSTANCE), NULL);
#else
          // supprime les & non grs comme raccourcis par GTK ( revoir)

          _TCHAR *ptr;
          while ((ptr = _tcschr(titleTmp, '&')))
            {
              memmove(ptr, &ptr[1], _tcslen(ptr) * sizeof(_TCHAR));
              ++ ptr;
            }

          hwndItem = gtk_button_new_with_label(titleTmp);
          gtk_box_pack_start(GTK_BOX(box), hwndItem, TRUE, TRUE, 0);

          static DATA data[3];
          if (itemNum <= 3)
            {
              data[itemNum - 1].id = itemId;
              data[itemNum - 1].hwnd = hwnd;

              g_signal_connect(hwndItem, "clicked", G_CALLBACK(MinDDialog_OnCommand), &data[itemNum - 1]);
            }
#endif

          if (hwndItem)
            {
              ++ result;
#ifndef MIND_GTK
              PostMessage(hwndItem, WM_SETFONT, (WPARAM)GetStockObject(DEFAULT_GUI_FONT), MAKELPARAM(TRUE, 0));
#else
#endif

              if (defaultButton == itemNum)
                {
#ifndef MIND_GTK
                  PostMessage(hwnd, DM_SETDEFID, (WPARAM) itemId, 0);
                  PostMessage(hwndItem, BM_SETSTYLE, (WPARAM)LOWORD(style), (LPARAM)TRUE);
#else
#endif
                }


#ifndef MIND_GTK
              // adapte la taille de la fentre

              RECT rClient;
              GetClientRect(hwndItem, &rClient);

              if (!width)
                {
                  int idealWidth = textSizeMaxW + (defaultWidth - rClient.right);
                  width = (idealWidth > defaultWidth) ? idealWidth : defaultWidth;
                }

              if (!height)
                {
                  int idealHeight = textSizeMaxH + (defaultHeight - rClient.bottom);
                  height = (idealHeight > defaultHeight) ? idealHeight : defaultHeight;
                }

              MoveWindow(hwndItem, pos.x, pos.y, width, height, TRUE);
#else
#endif

              if (horizontal)
                {
                  pos.x += width + 4;
                }
              else
                {
                  pos.y += height + 2;
                }
            }
        }
      while ((titlePtr = _tcschr(titlePtr, cSeparator)));

      free(title);
    }

  if (horizontal && horizontal[0])
    {
      free(horizontal);
    }

  return result;
}


/**
 * \fn int MinDDialog_CreateRadioGroup(HWND hwnd, _TCHAR *text, LONG_PTR itemId)
 * \brief Cration d'un groupe de cases d'option
 *
 * \param hwnd : handle de la boite de dialogue
 * \param text : texte de la commande
 * \param itemId : numro d'item
 *
 * \return le nombre de cases cres
 */
static int MinDDialog_CreateRadioGroup(HWND hwnd, _TCHAR *text, LONG_PTR itemId)
{
  int result = 0;

  POINT pos = MinDDialog_ExtractPoint(text, TEXT("AT"));
  _TCHAR * horizontal = MinDDialog_ExtractString(text, TEXT("HORZ"));
  int defaultButton = MinDDialog_ExtractInt(text, TEXT("DEFAULT"));

  // recherche et initialise la variable

  _TCHAR *variable = MinDDialog_ExtractVariable(text);
  if (variable && variable[0])
    {
      MinDVar *mindVar = MinDVariable_Retrieve(&mindVarGlobal, variable, 0);
      if (mindVar && mindVar->value)
        {
          defaultButton = _tcstol(mindVar->value, NULL, 10);
        }

      _TCHAR buffer[0x100];
      _sntprintf(buffer, sizeof(buffer) / sizeof(buffer[0]), TEXT("%d"), defaultButton);
      MinDVariable_Fill(&mindVarGlobal, 'N', variable, itemId, buffer);

      free(variable);
    }

  _TCHAR *title = MinDDialog_ExtractString(text, TEXT("TITLE"));

  if (title)
    {
      _TCHAR *titlePtr = title - 1;
      int itemNb = 1;

      while ((titlePtr = _tcschr(titlePtr + 1, cSeparator)))
        {
          ++ itemNb;
        }

      int width = MinDDialog_ExtractInt(text, TEXT("WIDTH"));
      int height = MinDDialog_ExtractInt(text, TEXT("HEIGHT"));

#ifndef MIND_GTK
      int defaultWidth = 24, defaultHeight = 24;

      width /= (horizontal ? itemNb : 1);
      height /= (horizontal ? 1 : itemNb);
#else
      GtkWidget * box = NULL;
#if GTK_MAJOR_VERSION > 2
      box = gtk_box_new(horizontal ? GTK_ORIENTATION_HORIZONTAL : GTK_ORIENTATION_VERTICAL, 2);
      gtk_box_set_homogeneous(GTK_BOX(box), TRUE);
#else
      box = horizontal ? gtk_hbox_new(TRUE, 2) : gtk_vbox_new(TRUE, 2);
#endif
      if (width && height)
        {
          gtk_widget_set_size_request(box, width, height);
        }

      gtk_fixed_put(GTK_FIXED(clientZone), box, pos.x, pos.y);
#endif

      // boucle de cration des cases

      titlePtr = title - 1;
      int itemNum = 0;

#ifndef MIND_GTK
      DWORD style = WS_CHILD | WS_VISIBLE | WS_TABSTOP | WS_GROUP | BS_AUTORADIOBUTTON;
#else
      GSList *group = NULL;
#endif

      do
        {
          ++ titlePtr;
          int length = _tcscspn(titlePtr, sSeparator);
          _TCHAR titleTmp[length + 1];
          _tcsncpy(titleTmp, titlePtr, length);
          titleTmp[length] = '\0';

          ++ itemId;
          ++ itemNum;

          HWND hwndItem = NULL;

#ifndef MIND_GTK
          hwndItem = CreateWindowEx(0, TEXT("BUTTON"), titleTmp, style,
                                    pos.x, pos.y, defaultWidth, defaultHeight,
                                    hwnd, (HMENU) itemId, (HINSTANCE)GetWindowLongPtr(hwnd, GWLP_HINSTANCE), NULL);

          style = WS_CHILD | WS_VISIBLE | WS_TABSTOP | BS_AUTORADIOBUTTON;
#else
          hwndItem = gtk_radio_button_new_with_label(group, titleTmp);
          gtk_box_pack_start(GTK_BOX(box), hwndItem, TRUE, TRUE, 0);

          group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(hwndItem));

          static DATA data[MAX_ITEM];
          int itemGtkNum = ITEM_INDEX_MASK & itemId;
          if (itemGtkNum <= MAX_ITEM)
            {
              data[itemGtkNum - 1].id = itemId;
              data[itemGtkNum - 1].hwnd = hwnd;

              g_signal_connect(hwndItem, "clicked", G_CALLBACK(MinDDialog_OnCommand), &data[itemGtkNum - 1]);
            }
#endif

          if (hwndItem)
            {
              ++ result;
#ifndef MIND_GTK
              PostMessage(hwndItem, WM_SETFONT, (WPARAM)GetStockObject(DEFAULT_GUI_FONT), MAKELPARAM(TRUE, 0));
#else
#endif

              if (defaultButton == itemNum)
                {
#ifndef MIND_GTK
                  PostMessage(hwndItem, BM_SETCHECK, (WPARAM)BST_CHECKED, (LPARAM)0);
#else
                  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(hwndItem), TRUE);
#endif
                }


#ifndef MIND_GTK
              // adapte la taille de la fentre

              RECT rClient;
              GetClientRect(hwndItem, &rClient);

              int textSizeX = rClient.right, textSizeY = rClient.bottom;
              int checkSizeX = defaultWidth, checkSizeY = defaultHeight;

              SIZE textSize;
              HDC hdc = GetDC(hwndItem);
              HFONT hfnt = SelectFont(hdc, GetStockObject(DEFAULT_GUI_FONT));

              GetTextExtentPoint32(hdc, titleTmp, _tcslen(titleTmp), &textSize);

              hfnt = SelectFont(hdc, hfnt);
              DeleteObject(hfnt);

              textSizeX = textSize.cx;
              textSizeY = textSize.cy;

              checkSizeX = GetSystemMetrics(SM_CXMENUCHECK);
              checkSizeY = GetSystemMetrics(SM_CYMENUCHECK);

              if (!width)
                {
                  int idealWidth = textSizeX + (defaultWidth - rClient.right) + checkSizeX;
                  width = idealWidth + (horizontal ? ((textSizeX < defaultWidth) ? textSizeX : defaultWidth) : 2);
                }

              if (!height)
                {
                  int idealHeight = textSizeY + (defaultHeight - rClient.bottom);
                  idealHeight = (checkSizeY > idealHeight) ? checkSizeY : idealHeight;
                  height = (idealHeight > defaultHeight) ? idealHeight : defaultHeight;
                }

              if (width && height)
                {
                  MoveWindow(hwndItem, pos.x, pos.y, width, height, TRUE);
                }
              else
                {
                  width = defaultWidth;
                  height = defaultHeight;
                }

              if (horizontal)
                {
                  pos.x += width;
                }
              else
                {
                  pos.y += height;
                }
#else
#endif
            }
        }
      while ((titlePtr = _tcschr(titlePtr, cSeparator)));

      if (title[0])
        {
          free(title);
        }
    }

  if (horizontal && horizontal[0])
    {
      free(horizontal);
    }

  return result;
}


/**
 * \fn BOOL MinDDialog_OnInitDialog(HWND hwnd, HWND hwndFocus, LPARAM lParam)
 * \brief Initialisation d'une bote de dialogue
 *
 * Cette fonction reoit le message WM_INITDIALOG et le traite.
 *
 * \param hwnd : handle de la boite de dialogue
 * \param hwndFocus : handle de la fentre recevant le focus
 * \param lParam : second paramtre du message
 *
 * \return TRUE (donne le focus  la fentre)
 */
static BOOL MinDDialog_OnInitDialog(HWND hwnd, HWND hwndFocus, LPARAM lParam)
{
#ifndef MIND_GTK
  PostMessage(hwnd, WM_SETFONT, (WPARAM)GetStockObject(DEFAULT_GUI_FONT), MAKELPARAM(TRUE, 0));
#else
#endif

  _TCHAR *line = (_TCHAR *) lParam;

  LONG_PTR itemId = 0;
  int width = 0;
  int height = 0;

  // rglage de la bote de dialogue

  _TCHAR *item = MinDDialog_LocateItem(line, TEXT("DIALOG"));

  if (item)
    {
      // dfinition du titre de la bote de dialogue

      _TCHAR *title = MinDDialog_ExtractString(item, TEXT("TITLE"));

      if (title && title[0])
        {
          SetWindowText(hwnd, title);
          free(title);
        }
      else
        {
          SetWindowText(hwnd, TEXT(""));
        }

      // dfinition des dimensions de la bote de dialogue

      width = MinDDialog_ExtractInt(item, TEXT("WIDTH"));
      height = MinDDialog_ExtractInt(item, TEXT("HEIGHT"));
    }
  else
    {
      SetWindowText(hwnd, TEXT(""));
    }

#ifndef MIND_GTK
  int defaultWidth = 235, defaultHeight = 110;

  MoveWindow(hwnd, 0, 0, defaultWidth, defaultHeight, TRUE);

  RECT rWindow, rClient;

  GetWindowRect(hwnd, &rWindow);
  GetClientRect(hwnd, &rClient);

  int xMax = 0, yMax = 0;
  int xBase = rWindow.right - rClient.right;
  int yBase = rWindow.bottom - rClient.bottom;
#else
  gtk_window_set_position(GTK_WINDOW(hwnd), GTK_WIN_POS_CENTER);
  if (width && height)
    {
      gtk_window_set_default_size(GTK_WINDOW(hwnd), width, height);
    }
#endif


  // cration des contrles statiques

  itemId = STATIC_ID;

  item = line;
  while ((item = MinDDialog_LocateItem(item, TEXT("TEXT"))))
    {
      ++ itemId;
      MinDDialog_CreateStatic(hwnd, item, itemId);

#ifndef MIND_GTK
      GetWindowRect(GetDlgItem(hwnd, itemId), &rWindow);

      if (rWindow.right > xMax)
        {
          xMax = rWindow.right;
        }

      if (rWindow.bottom > yMax)
        {
          yMax = rWindow.bottom;
        }
#else
#endif
    }


  // cration des contrles d'dition

  itemId = EDITBOX_ID;

  item = line;
  while ((item = MinDDialog_LocateItem(item, TEXT("EDIT"))))
    {
      ++ itemId;
      MinDDialog_CreateEditBox(hwnd, item, itemId);

#ifndef MIND_GTK
      GetWindowRect(GetDlgItem(hwnd, itemId), &rWindow);

      if (rWindow.right > xMax)
        {
          xMax = rWindow.right;
        }

      if (rWindow.bottom > yMax)
        {
          yMax = rWindow.bottom;
        }
#else
#endif
    }


  // cration (dgrade) des items (rfrences externes insolubles)
  // sous la forme de contrles d'dition

  item = line;
  while ((item = MinDDialog_LocateItem(item, TEXT("ITEM"))))
    {
      ++ itemId;
      MinDDialog_CreateEditBox(hwnd, item, itemId);

#ifndef MIND_GTK
      GetWindowRect(GetDlgItem(hwnd, itemId), &rWindow);

      if (rWindow.right > xMax)
        {
          xMax = rWindow.right;
        }

      if (rWindow.bottom > yMax)
        {
          yMax = rWindow.bottom;
        }
#else
#endif
    }


  // cration des cases  cocher

  itemId = CHECKBOX_ID;

  item = line;
  while ((item = MinDDialog_LocateItem(item, TEXT("CHECKBOX"))))
    {
      ++ itemId;
      MinDDialog_CreateCheckBox(hwnd, item, itemId);

#ifndef MIND_GTK
      GetWindowRect(GetDlgItem(hwnd, itemId), &rWindow);

      if (rWindow.right > xMax)
        {
          xMax = rWindow.right;
        }

      if (rWindow.bottom > yMax)
        {
          yMax = rWindow.bottom;
        }
#else
#endif
    }


  // cration des listes droulantes

  itemId = DROPBOX_ID;

  item = line;
  while ((item = MinDDialog_LocateItem(item, TEXT("DROPDOWN"))))
    {
      ++ itemId;
      MinDDialog_CreateDropBox(hwnd, item, itemId);

#ifndef MIND_GTK
      GetWindowRect(GetDlgItem(hwnd, itemId), &rWindow);

      if (rWindow.right > xMax)
        {
          xMax = rWindow.right;
        }

      if (rWindow.bottom > yMax)
        {
          yMax = rWindow.bottom;
        }
#else
#endif
    }


  // cration des cases d'option

  itemId = RADIO_ID;

  item = line;
  while ((item = MinDDialog_LocateItem(item, TEXT("RADIOBUTTON"))))
    {
      itemId += 0x100;
      int buttonNb = MinDDialog_CreateRadioGroup(hwnd, item, itemId);

      while (buttonNb)
        {
#ifndef MIND_GTK
          GetWindowRect(GetDlgItem(hwnd, itemId + buttonNb), &rWindow);

          if (rWindow.right > xMax)
            {
              xMax = rWindow.right;
            }

          if (rWindow.bottom > yMax)
            {
              yMax = rWindow.bottom;
            }
#else
#endif
          -- buttonNb;
        }
    }


  // cration des packs de boutons

  itemId = BUTTON_ID;

  item = line;
  while ((item = MinDDialog_LocateItem(item, TEXT("BUTTONSET"))))
    {
      itemId += 0x100;
      int buttonNb = MinDDialog_CreateButtonSet(hwnd, item, itemId);

      while (buttonNb)
        {
#ifndef MIND_GTK
          GetWindowRect(GetDlgItem(hwnd, itemId + buttonNb), &rWindow);

          if (rWindow.right > xMax)
            {
              xMax = rWindow.right;
            }

          if (rWindow.bottom > yMax)
            {
              yMax = rWindow.bottom;
            }
#else
#endif
          -- buttonNb;
        }
    }


#ifndef MIND_GTK
  // ajustement

  xMax += 8 - xBase;
  yMax += 8 - yBase;

  // rglage des dimensions de la bote de dialogue

  if (!width)
    {
      width = (xMax > defaultWidth) ? xMax : defaultWidth;
    }

  if (!height)
    {
      height = (yMax > defaultHeight) ? yMax : defaultHeight;
    }

  MoveWindow(hwnd, 0, 0, width, height, FALSE);
  GetClientRect(hwnd, &rClient);

  width += (width - rClient.right);
  height += (height - rClient.bottom);

  MoveWindow(hwnd, 0, 0, width, height, FALSE);


  // centrage de la bote de dialogue

  RECT rScreen;

  HWND hwndOwner = GetWindowOwner(hwnd);
  if (!hwndOwner)
    {
      hwndOwner = GetDesktopWindow();
    }

  GetWindowRect(hwndOwner, &rScreen);
  GetWindowRect(hwnd, &rWindow);

  rWindow.left = ((rScreen.right + rScreen.left) - (rWindow.right - rWindow.left)) / 2;
  rWindow.top = ((rScreen.bottom + rScreen.top) - (rWindow.bottom - rScreen.top)) / 2;

  MoveWindow(hwnd, rWindow.left, rWindow.top, width, height, FALSE);
#else
#endif


  // fin d'initialisation

#ifndef MIND_GTK
  SetForegroundWindow(hwnd);
#else
  gtk_widget_show_all(hwnd);
#endif

  return TRUE;
}


/**
 * \fn void MinDDialog_OnClose(HWND hwnd)
 * \brief Fermeture d'une boite de dialogue MinD
 *
 * Cette fonction reoit le message WM_CLOSE
 *
 * \param hwnd : handle de la boite de dialogue
 */
#ifndef MIND_GTK
static void MinDDialog_OnClose(HWND hwnd)
{
  PostMessage(hwnd, WM_COMMAND, MAKEWPARAM(BUTTON_ID + 2, 0), (LPARAM) GetDlgItem(hwnd, BUTTON_ID + 2));
}
#else
#endif


/**
 * \fn void MinDDialog_OnCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
 * \brief Gestion des commandes du dialogue MinD
 *
 * \param hwnd : handle de la boite de dialogue
 * \param id : ID de la commande
 * \param hwndCtl : handle du contrle
 * \param codeNotify : code dfini par le contrle
 */
#ifndef MIND_GTK
static void MinDDialog_OnCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{
#else
static void MinDDialog_OnCommand(HWND hwndCtl, DATA *data)
{
  GtkWidget *hwnd = data->hwnd;
  int id = data->id;
#endif
  int type = ITEM_TYPE_MASK & id;
  int group = ITEM_GROUP_MASK & id;
  int index = ITEM_INDEX_MASK & id;

  switch (type)
    {
      case BUTTON_ID :
        {
          EndDialog(hwnd, index - 1);
          break;
        }

      case EDITBOX_ID :
        {
          int length = 0;
#ifndef MIND_GTK
          length = GetWindowTextLength(hwndCtl);
#else
          const gchar *g_buffer = gtk_entry_get_text(GTK_ENTRY(hwndCtl));
          length = _tcslen(g_buffer);
#endif
          _TCHAR *buffer = calloc(sizeof(_TCHAR), length + 1);
          if (buffer)
            {
#ifndef MIND_GTK
              GetWindowText(hwndCtl, buffer, length + 1);
#else
              _tcsncpy(buffer, g_buffer, length);
#endif
              MinDVariable_Fill(&mindVarGlobal, 'C', NULL, id, buffer);
              free(buffer);
            }

          break;
        }

      case DROPBOX_ID :
        {
          int length = 0;
#ifndef MIND_GTK
          length = GetWindowTextLength(hwndCtl);
#else
          gchar *g_buffer = gtk_combo_box_text_get_active_text(GTK_COMBO_BOX_TEXT(hwndCtl));
          length = _tcslen(g_buffer);
#endif
          _TCHAR *buffer = calloc(sizeof(_TCHAR), length + 1);
          if (buffer)
            {
#ifndef MIND_GTK
              GetWindowText(hwndCtl, buffer, length + 1);
#else
              _tcsncpy(buffer, g_buffer, length);
              g_free(g_buffer);
#endif
              MinDVariable_Fill(&mindVarGlobal, 'C', NULL, id, buffer);
              free(buffer);
            }

          break;
        }

      case RADIO_ID :
        {
          _TCHAR buffer[0x100] = TEXT("");
          _sntprintf(buffer, sizeof(buffer) / sizeof(buffer[0]), TEXT("%d"), index);
          MinDVariable_Fill(&mindVarGlobal, 'N', NULL, group, buffer);

          break;
        }

      case CHECKBOX_ID :
        {
          BOOL test = FALSE;
#ifndef MIND_GTK
          test = (IsDlgButtonChecked(hwnd, id) == BST_CHECKED) ? TRUE : FALSE;
#else
          test = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(hwndCtl));
#endif
          if (test)
            {
              MinDVariable_Fill(&mindVarGlobal, 'L', NULL, id, TEXT("TRUE"));
            }
          else
            {
              MinDVariable_Fill(&mindVarGlobal, 'L', NULL, id, TEXT("FALSE"));
            }

          break;
        }

      default :
        {
          break;
        }
    }
}


/**
 * \fn BOOL CALLBACK MinDDialog_Procedure(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
 * \brief affiche un dialogue  partir d'une commande DIALOG
 *
 * \param hwnd : handle de la boite de dialogue
 * \param uMsg : message  traiter
 * \param wParam : premier paramtre du message
 * \param lParam : second paramtre du message
 */
#ifndef MIND_GTK
BOOL CALLBACK MinDDialog_Procedure(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
  switch (uMsg)
    {
        HANDLE_MSG(hwnd, WM_INITDIALOG, MinDDialog_OnInitDialog);
        HANDLE_MSG(hwnd, WM_CLOSE, MinDDialog_OnClose);
        HANDLE_MSG(hwnd, WM_COMMAND, MinDDialog_OnCommand);

      default :
        {
          break;
        }
    }

  return FALSE;
}
#else
int MinDDialog_Procedure(HWND hwndParent, _TCHAR *line)
{
  int result = 0;

  GtkWidget *hwnd = gtk_dialog_new_with_buttons (TEXT(""),
                                                 GTK_WINDOW(hwndParent),
                                                 GTK_DIALOG_MODAL,
                                                 NULL, NULL);

  // les lignes suivantes permettent de vider la bote de dialogue
  // de son contenu, notamment la GtkBox par dfaut, afin de la remplacer
  // par une zone fixe et de placer prcisment des contrles.

  GList *content = gtk_container_get_children(GTK_CONTAINER(hwnd));

  while (content)
    {
      gtk_widget_destroy(content->data);
      content = g_list_next(content);
    }

  g_list_free(content);

  clientZone = gtk_fixed_new();
  gtk_container_set_border_width(GTK_CONTAINER(clientZone), 4);
  gtk_container_add(GTK_CONTAINER(hwnd), clientZone);

  MinDDialog_OnInitDialog(hwnd, NULL, line);

  gtk_widget_show_all(hwnd);
  result = gtk_dialog_run(GTK_DIALOG(hwnd));
  gtk_widget_destroy(hwnd);

  if (result < 0)
    {
      result = 1;
    }

  return result;
}
#endif

