/**
 * \file input.c
 *
 * \brief Implmentation des fonctions et macros pour la gestion des dialogues simplifis
 * \author Denis MARTIN
 * \date 2017-2019
 */


/*

Copyright (C) 2017, 2019 Denis MARTIN

---------------------------------------

This file is part of MinD.

MinD is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or any later version.

MinD is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with MinD.  If not, see <http://www.gnu.org/licenses/>.

--------------------------------------------------------------------------

Ce fichier fait partie de MinD.

MinD est un logiciel libre : il peut tre redistribu et/ou modifi
selon les termes de la Licence Publique Gnrale GNU telle que publie par
la Free Software Foundation, soit la version 3 de la licence ou toute
version ultrieure.

MinD est distribu dans l'espoir qu'il sera utile,
mais SANS AUCUNE GARANTIE ; sans mme la garantie implicite de
VALEUR MARCHANDE ou d'ADEQUATION AVEC UN BUT PARTICULIER. Voir la
Licence Publique Gnrale GNU pour plus de dtails.

Une copie de la Licence Publique Gnrale GNU devrait tre fournie
avec MinD. Si ce n'est pas le cas, voir <http://www.gnu.org/licenses/>.

*/


#include "input.h"


#ifndef MIND_GTK
#else
static void MinDInput_OnCommand(HWND hwndCtl, DATA *data);
#endif


/**
 * \fn _TCHAR *MinDInput_ExtractVariable(_TCHAR *text)
 * \brief extrait le nom d'une variable d'une commande de dialogue simplifi
 *
 * la fonction est appele avec un pointeur juste aprs le mot-clef "TO",
 * chane dont elle extrait le nom qui suit (dans ou hors guillemets).
 *
 * \param text : pointeur juste aprs "TO" dans le texte de la commande
 *
 * \return une chane qui doit tre libre par l'appelant
 * \return NULL si aucun argument n'est trouv
 */
static _TCHAR *MinDInput_ExtractVariable(_TCHAR *text)
{
  _TCHAR *result = RetrieveValueInString(text, NULL);

  if (result && !result[0])
    {
      result = NULL;
    }

  return result;
}


/**
 * \fn _TCHAR *MinDInput_ExtractTitle(_TCHAR *text)
 * \brief extrait une chane d'une commande de dialogue simplifi
 *
 * la fonction parcourt la commande jusqu' rencontrer le mot-clef "TO" puis extrait la chane qui la prcde,
 * sauf si celle-ci est prcde du mot-clef FIELDS, auquel cas c'est la chane situe encore avant qui est extraite.
 *
 * \param text : texte de la commande
 *
 * \return une chane qui doit tre libre par l'appelant
 * \return NULL si aucun argument non vide n'est trouv
 */
static _TCHAR *MinDInput_ExtractTitle(_TCHAR *text)
{
  _TCHAR *textNext = text;
  _TCHAR *result = RetrieveValueInString(text, &textNext);

  if (result && result[0])
    {
      result = RetrieveValueInString(textNext, NULL);
    }

  if (result && !result[0])
    {
      result = NULL;
    }

  return result;
}


/**
 * \fn _TCHAR *MinDInput_LocateItem(_TCHAR *text)
 * \brief trouve un item dans une commande de dialogue simplifi
 *
 * la fonction parcourt la commande jusqu' rencontrer le mot-clef "TO"
 *
 * \param text : texte de la commande
 *
 * \return pointeur sur le caractre suivant le mot-clef "TO"
 * \return NULL si l'item n'est pas trouv
 */
static _TCHAR *MinDInput_LocateItem(_TCHAR *text)
{
  _TCHAR *result = LocateKeywordInString(text, TEXT("TO"), NULL, NULL);

  return result;
}


/**
 * \fn BOOL MinDInput_CreateStatic(HWND hwnd, _TCHAR *text, LONG_PTR itemId)
 * \brief Cration d'un contrle statique (texte)
 *
 * \param hwnd : handle de la boite de dialogue
 * \param text : texte de la commande
 * \param itemId : numro d'item
 *
 * \return TRUE si tout s'est bien pass
 * \return FALSE dans le cas contraire
 */
static BOOL MinDInput_CreateStatic(HWND hwnd, _TCHAR *text, LONG_PTR itemId)
{
  BOOL result = FALSE;

  _TCHAR *title = MinDInput_ExtractTitle(text);
  if (!title)
    {
      title = TEXT("");
    }

  HWND hwndItem = NULL;

#ifndef MIND_GTK
  int defaultX = 5, defaultY = 5;
  int defaultWidth = 210, defaultHeight = 22;

  int itemNum = ITEM_INDEX_MASK & itemId;

  hwndItem = CreateWindowEx(0, TEXT("STATIC"), title, WS_CHILD | WS_VISIBLE | WS_TABSTOP,
                            defaultX, defaultY + (defaultHeight + defaultY) * (itemNum - 1),
                            defaultWidth, defaultHeight + defaultY,
                            hwnd, (HMENU) itemId, (HINSTANCE)GetWindowLongPtr(hwnd, GWLP_HINSTANCE), NULL);
#else
  hwndItem = gtk_label_new(title);
  gtk_box_pack_start(GTK_BOX(hwnd), hwndItem, TRUE, TRUE, 0);
#endif

  if (hwndItem)
    {
      result = TRUE;
#ifndef MIND_GTK
      PostMessage(hwndItem, WM_SETFONT, (WPARAM)GetStockObject(DEFAULT_GUI_FONT), MAKELPARAM(TRUE, 0));
#else
#endif

      // pas d'adaptation de la taille de la fentre
    }

  if (title && title[0])
    {
      free(title);
    }

  return result;
}


/**
 * \fn BOOL MinDInput_CreateEditBox(HWND hwnd, _TCHAR *text, LONG_PTR itemId)
 * \brief Cration d'un contrle d'dition
 *
 * \param hwnd : handle de la boite de dialogue (API Win32)
 *               ou bote horizontale de l'item (GTK)
 * \param text : texte de la commande
 * \param itemId : numro d'item
 *
 * \return TRUE si tout s'est bien pass
 * \return FALSE dans le cas contraire
 */
static BOOL MinDInput_CreateEditBox(HWND hwnd, _TCHAR *text, LONG_PTR itemId)
{
  BOOL result = FALSE;

  _TCHAR *title = TEXT("");


  // initialise la variable correspondante

  _TCHAR *variable = MinDInput_ExtractVariable(text);
  if (variable && variable[0])
    {
      MinDVar *mindVar = MinDVariable_Retrieve(&mindVarGlobal, variable, 0);
      if (mindVar && mindVar->value)
        {
          if (mindVar->value[0])
            {
              title = _tcsdup(mindVar->value);
            }

          if (!title)
            {
              title = TEXT("");
            }
        }

      MinDVariable_Fill(&mindVarGlobal, 'C', variable, itemId, title);
      free(variable);
    }

  int itemNum = ITEM_INDEX_MASK & itemId;

  HWND hwndItem = NULL;

  int defaultWidth = 290, defaultHeight = 22;

#ifndef MIND_GTK
  int defaultX = 215, defaultY = 5;

  hwndItem = CreateWindowEx(WS_EX_CLIENTEDGE,
                            TEXT("EDIT"), title, WS_CHILD | WS_VISIBLE | WS_TABSTOP | ES_AUTOHSCROLL,
                            defaultX, defaultY + (defaultHeight + defaultY) * (itemNum - 1),
                            defaultWidth, defaultHeight,
                            hwnd, (HMENU) itemId, (HINSTANCE)GetWindowLongPtr(hwnd, GWLP_HINSTANCE), NULL);
#else

  hwndItem = gtk_entry_new();

  gtk_box_pack_end(GTK_BOX(hwnd), hwndItem, TRUE, TRUE, 0);

  gtk_widget_set_size_request(hwndItem, defaultWidth, defaultHeight);
  gtk_entry_set_text(GTK_ENTRY(hwndItem), title);

  static DATA data[MAX_ITEM];
  if (itemNum <= MAX_ITEM)
    {
      data[itemNum - 1].id = itemId;
      data[itemNum - 1].hwnd = hwnd;

      g_signal_connect(hwndItem, "changed", G_CALLBACK(MinDInput_OnCommand), &data[itemNum - 1]);
    }
#endif

  if (hwndItem)
    {
      result = TRUE;
#ifndef MIND_GTK
      PostMessage(hwndItem, WM_SETFONT, (WPARAM)GetStockObject(DEFAULT_GUI_FONT), MAKELPARAM(TRUE, 0));
#else
#endif

      // pas d'adaptation de la taille de la fentre
    }

  if (title && title[0])
    {
      free(title);
    }

  return result;
}


/**
 * \fn int MinDInput_CreateButtonSet(HWND hwnd, LONG_PTR itemId)
 * \brief Cration d'un bloc de deux boutons (OK et Abandon)
 *
 * \param hwnd : handle de la boite de dialogue
 * \param itemId : numro d'item
 *
 * \return le nombre de boutons crs (normalement 2)
 */
static int MinDInput_CreateButtonSet(HWND hwnd, LONG_PTR itemId)
{
  int result = 0;

  int defaultButton = 1;
  _TCHAR *title = TEXT("OK;Abandon");

#ifndef MIND_GTK
  int horizontal = 1;

  int defaultWidth = 80, defaultHeight = 24;
  int textSizeMaxW = 0, textSizeMaxH = 0;
#else
#endif

  _TCHAR *titlePtr = title - 1;
  int buttonNb = 0;

  do
    {
      ++ titlePtr;
      int length = _tcscspn(titlePtr, sSeparator);
      if (length)
        {
#ifndef MIND_GTK
          SIZE textSize;
          ZeroMemory(&textSize, sizeof(textSize));
          HDC hdc = GetDC(hwnd);
          HFONT hfnt = SelectFont(hdc, GetStockObject(DEFAULT_GUI_FONT));

          GetTextExtentPoint32(hdc, titlePtr, length, &textSize);

          hfnt = SelectFont(hdc, hfnt);
          DeleteObject(hfnt);

          textSizeMaxW = (textSize.cx > textSizeMaxW) ? textSize.cx : textSizeMaxW;
          textSizeMaxH = (textSize.cy > textSizeMaxH) ? textSize.cy : textSizeMaxH;
#else
#endif
        }

      ++ buttonNb;
    }
  while ((titlePtr = _tcschr(titlePtr, cSeparator)));

  // boucle de cration des boutons

  titlePtr = title - 1;
  int buttonNum = 0;

  do
    {
      ++ titlePtr;
      int length = _tcscspn(titlePtr, sSeparator);
      _TCHAR titleTmp[length + 1];
      _tcsncpy(titleTmp, titlePtr, length);
      titleTmp[length] = '\0';

      ++ itemId;
      ++ buttonNum;

#ifndef MIND_GTK
      DWORD style = WS_CHILD | WS_VISIBLE | WS_TABSTOP;
#else
#endif
      if (defaultButton == buttonNum)
        {
#ifndef MIND_GTK
          style |= BS_DEFPUSHBUTTON;
#else
#endif
        }

      HWND hwndItem = NULL;

#ifndef MIND_GTK
      hwndItem = CreateWindowEx(0, TEXT("BUTTON"), titleTmp, style, 0, 0, defaultWidth, defaultHeight,
                                hwnd, (HMENU) itemId, (HINSTANCE)GetWindowLongPtr(hwnd, GWLP_HINSTANCE), NULL);
#else
      // supprime les &, non grs comme raccourcis par GTK ( revoir)

      _TCHAR *ptr;
      while ((ptr = _tcschr(titleTmp, '&')))
        {
          memmove(ptr, &ptr[1], _tcslen(ptr) * sizeof(_TCHAR));
          ++ ptr;
        }

      hwndItem = gtk_dialog_add_button(GTK_DIALOG(hwnd), titleTmp, buttonNum - 1);
#endif

      if (hwndItem)
        {
          ++ result;
#ifndef MIND_GTK
          PostMessage(hwndItem, WM_SETFONT, (WPARAM)GetStockObject(DEFAULT_GUI_FONT), MAKELPARAM(TRUE, 0));
#else
#endif

          if (defaultButton == buttonNum)
            {
#ifndef MIND_GTK
              PostMessage(hwnd, DM_SETDEFID, (WPARAM) itemId, 0);
              PostMessage(hwndItem, BM_SETSTYLE, (WPARAM)LOWORD(style), (LPARAM)TRUE);
#else
#endif
            }


#ifndef MIND_GTK
          // adapte la taille du bouton

          RECT rClient;
          GetClientRect(hwndItem, &rClient);

          int idealWidth = textSizeMaxW + (defaultWidth - rClient.right);
          int width = (idealWidth > defaultWidth) ? idealWidth : defaultWidth;

          int idealHeight = textSizeMaxH + (defaultHeight - rClient.bottom);
          int height = (idealHeight > defaultHeight) ? idealHeight : defaultHeight;

          RECT rDialog;
          GetClientRect(hwnd, &rDialog);

          MoveWindow(hwndItem,
                     rDialog.right - (width + 4) * (horizontal ? (buttonNb - (buttonNum - 1)) : 1),
                     rDialog.bottom + 8 + (height + 2) * (horizontal ? 0 : buttonNum),
                     width, height, TRUE);
#else
#endif
        }
    }
  while ((titlePtr = _tcschr(titlePtr, cSeparator)));

  return result;
}


/**
 * \fn BOOL MinDInput_OnInitDialog(HWND hwnd, HWND hwndFocus, LPARAM lParam)
 * \brief Initialisation d'une bote de dialogue simplifie
 *
 * Cette fonction reoit le message WM_INITDIALOG et le traite.
 *
 * \param hwnd : handle de la boite de dialogue
 * \param hwndFocus : handle de la fentre recevant le focus
 * \param lParam : second paramtre du message
 *
 * \return TRUE (donne le focus  la fentre)
 */
static BOOL MinDInput_OnInitDialog(HWND hwnd, HWND hwndFocus, LPARAM lParam)
{
#ifndef MIND_GTK
  PostMessage(hwnd, WM_SETFONT, (WPARAM)GetStockObject(DEFAULT_GUI_FONT), MAKELPARAM(TRUE, 0));
#else
#endif

  _TCHAR *line = (_TCHAR *) lParam;

  LONG_PTR itemId = 0;

  int defaultWidth = 235, defaultHeight = 110;

#ifndef MIND_GTK
  RECT rWindow, rClient;
  MoveWindow(hwnd, 0, 0, defaultWidth, defaultHeight, TRUE);

  GetWindowRect(hwnd, &rWindow);
  GetClientRect(hwnd, &rClient);

  int xMax = 0, yMax = 0;
  int xBase = rWindow.right - rClient.right;
  int yBase = rWindow.bottom - rClient.bottom;
#else
  gtk_window_set_position(GTK_WINDOW(hwnd), GTK_WIN_POS_CENTER);
  gtk_window_set_default_size(GTK_WINDOW(hwnd), defaultWidth, defaultHeight);

  GtkWidget *hwndDialog = hwnd;

  GtkWidget *box = gtk_dialog_get_content_area(GTK_DIALOG(hwndDialog));
  gtk_container_set_border_width(GTK_CONTAINER(box), 4);
#endif


  // cration des contrles statiques

  itemId = STATIC_ID;

  _TCHAR *item = line;
  _TCHAR *nextItem = NULL;

  while ((nextItem = MinDInput_LocateItem(item)))
    {

#ifndef MIND_GTK
#else
#if GTK_MAJOR_VERSION > 2
      hwnd = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 0);
      gtk_box_set_homogeneous(GTK_BOX(hwnd), TRUE);
#else
      hwnd = gtk_hbox_new(TRUE, 0);
#endif

      gtk_box_pack_start(GTK_BOX(box), hwnd, TRUE, TRUE, 2);
#endif

      ++ itemId;
      MinDInput_CreateStatic(hwnd, item, itemId);

      item = nextItem;

      // cration du contrle d'dition correspondant

      itemId += (EDITBOX_ID - STATIC_ID);
      MinDInput_CreateEditBox(hwnd, item, itemId);

#ifndef MIND_GTK
      GetWindowRect(GetDlgItem(hwnd, itemId), &rWindow);

      if (rWindow.right > xMax)
        {
          xMax = rWindow.right;
        }

      if (rWindow.bottom > yMax)
        {
          yMax = rWindow.bottom;
        }
#else
#endif

      itemId += (STATIC_ID - EDITBOX_ID);
    }


#if 0
  // cration des contrles d'dition

  itemId = EDITBOX_ID;

  item = line;
  while ((item = MinDInput_LocateItem(item)))
    {
      ++ itemId;
      MinDInput_CreateEditBox(hwnd, item, itemId);

      GetWindowRect(GetDlgItem(hwnd, itemId), &rWindow);

      if (rWindow.right > xMax)
        {
          xMax = rWindow.right;
        }

      if (rWindow.bottom > yMax)
        {
          yMax = rWindow.bottom;
        }
    }
#endif


#ifndef MIND_GTK
  // ajustement

  xMax += 8 - xBase;
  yMax += 8 - yBase;

  // premier rglage des dimensions de la bote de dialogue

  int width = (xMax > defaultWidth) ? xMax : defaultWidth;
  int height = (yMax > defaultHeight) ? yMax : defaultHeight;

  MoveWindow(hwnd, 0, 0, width, height, FALSE);
  GetClientRect(hwnd, &rClient);

  width += (width - rClient.right);
  height += (height - rClient.bottom);

  MoveWindow(hwnd, 0, 0, width, height, FALSE);
#else
  hwnd = hwndDialog;
#endif


  // cration du pack de boutons

  itemId = BUTTON_ID;
  int buttonNb = MinDInput_CreateButtonSet(hwnd, itemId);

  while (buttonNb)
    {
#ifndef MIND_GTK
      GetWindowRect(GetDlgItem(hwnd, itemId + buttonNb), &rWindow);

      if (rWindow.right > xMax)
        {
          xMax = rWindow.right;
        }

      if (rWindow.bottom > yMax)
        {
          yMax = rWindow.bottom;
        }
#else
#endif
      -- buttonNb;
    }


#ifndef MIND_GTK
  // ajustement

  xMax += 8 - xBase;
  yMax += 8 - yBase;


  // rglage dfinitif des dimensions de la bote de dialogue

  width = (xMax > defaultWidth) ? xMax : defaultWidth;
  height = (yMax > defaultHeight) ? yMax : defaultHeight;

  MoveWindow(hwnd, 0, 0, width, height, FALSE);
  GetClientRect(hwnd, &rClient);

  width += (width - rClient.right);
  height += (height - rClient.bottom);

  MoveWindow(hwnd, 0, 0, width, height, FALSE);


  // centrage de la bote de dialogue

  RECT rScreen;

  HWND hwndOwner = GetWindowOwner(hwnd);
  if (!hwndOwner)
    {
      hwndOwner = GetDesktopWindow();
    }

  GetWindowRect(hwndOwner, &rScreen);
  GetWindowRect(hwnd, &rWindow);

  rWindow.left = ((rScreen.right + rScreen.left) - (rWindow.right - rWindow.left)) / 2;
  rWindow.top = ((rScreen.bottom + rScreen.top) - (rWindow.bottom - rScreen.top)) / 2;

  MoveWindow(hwnd, rWindow.left, rWindow.top, width, height, FALSE);
#else
#endif


  // fin d'initialisation

#ifndef MIND_GTK
  SetForegroundWindow(hwnd);
#else
  gtk_widget_show_all(hwnd);
#endif

  return TRUE;
}


/**
 * \fn void MinDInput_OnClose(HWND hwnd)
 * \brief Fermeture d'une boite de dialogue MinD
 *
 * Cette fonction reoit le message WM_CLOSE
 *
 * \param hwnd : handle de la boite de dialogue
 */
#ifndef MIND_GTK
static void MinDInput_OnClose(HWND hwnd)
{
  PostMessage(hwnd, WM_COMMAND, MAKEWPARAM(BUTTON_ID + 2, 0), (LPARAM) GetDlgItem(hwnd, BUTTON_ID + 2));
}
#else
#endif


/**
 * \fn void MinDInput_OnCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
 * \brief Gestion des commandes du dialogue MinD
 *
 * \param hwnd : handle de la boite de dialogue
 * \param id : ID de la commande
 * \param hwndCtl : handle du contrle
 * \param codeNotify : code dfini par le contrle
 */
#ifndef MIND_GTK
static void MinDInput_OnCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{
#else
static void MinDInput_OnCommand(HWND hwndCtl, DATA *data)
{
  GtkWidget *hwnd = data->hwnd;
  int id = data->id;
#endif
  int type = ITEM_TYPE_MASK & id;
  //int group = (ITEM_GROUP_MASK & id) > 8;
  int index = ITEM_INDEX_MASK & id;

  switch (type)
    {
      case BUTTON_ID :
        {
          EndDialog(hwnd, index - 1);
          break;
        }

      case EDITBOX_ID :
        {
          int length = 0;
#ifndef MIND_GTK
          length = GetWindowTextLength(hwndCtl);
#else
          const gchar *g_buffer = gtk_entry_get_text(GTK_ENTRY(hwndCtl));
          length = _tcslen(g_buffer);
#endif
          _TCHAR *buffer = calloc(sizeof(_TCHAR), length + 1);
          if (buffer)
            {
#ifndef MIND_GTK
              GetWindowText(hwndCtl, buffer, length + 1);
#else
              _tcsncpy(buffer, g_buffer, length);
#endif
              MinDVariable_Fill(&mindVarGlobal, 'C', NULL, id, buffer);
              free(buffer);
            }

          break;
        }

      default :
        {
          break;
        }
    }
}


/**
 * \fn BOOL CALLBACK MinDInput_Procedure(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
 * \brief affiche un dialogue simplifi  partir d'une commande ACCEPT
 *
 * \param hwnd : handle de la boite de dialogue
 * \param uMsg : message
 * \param wParam : premier paramtre du message
 * \param lParam : second paramtre du message
 */
#ifndef MIND_GTK
BOOL CALLBACK MinDInput_Procedure(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
  switch (uMsg)
    {
        HANDLE_MSG(hwnd, WM_INITDIALOG, MinDInput_OnInitDialog);
        HANDLE_MSG(hwnd, WM_CLOSE, MinDInput_OnClose);
        HANDLE_MSG(hwnd, WM_COMMAND, MinDInput_OnCommand);

      default :
        {
          break;
        }
    }

  return FALSE;
}
#else
int MinDInput_Procedure(HWND hwndParent, _TCHAR *line)
{
  int result = 0;

  GtkWidget *hwnd = gtk_dialog_new_with_buttons (TEXT("Remplissez les valeurs attendues"),
                                                 GTK_WINDOW(hwndParent),
                                                 GTK_DIALOG_MODAL,
                                                 NULL, NULL);

  MinDInput_OnInitDialog(hwnd, NULL, line);
  gtk_window_set_position(GTK_WINDOW(hwnd), GTK_WIN_POS_CENTER);

  gtk_widget_show_all(hwnd);
  result = gtk_dialog_run(GTK_DIALOG(hwnd));
  gtk_widget_destroy(hwnd);

  if (result < 0)
    {
      result = 1;
    }

  return result;
}
#endif

