/**
 * \file mind.c
 *
 * \brief Implmentation de la procdure de fentre et de l'interface
 *
 * \author Denis MARTIN
 * \date 2017-2019
 *
 */


/**
 * \mainpage MinD 1.1.2

### Pourquoi MinD ?

MinD fournit un outil lger  divers langages de script
pour mettre en oeuvre simplement l'interface graphique native
de Windows ou l'interface GTK sous Linux.

MinD prend en entre un fichier texte dcrivant un dialogue
et gnre en sortie un fichier texte attribuant aux variables
les valeurs choisies par l'utilisateur.

#### Exemple d'une bote de dialogue simple :

 * \code{.unparsed}

DIALOG (BUTTONSET AT 160 120) (EDIT TO "VariableEdit" AT 20 50 DEFAULT "EditBox")

 * \endcode

Ce descriptif produit la bote de dialogue suivante :

 * \image html MinD_simple.png

#### Exemple d'une bote de dialogue complexe prsentant les diffrents contrles possibles :

 * \code{.unparsed}

DIALOG (DIALOG TITLE "DialogBox 2" WIDTH 271 HEIGHT 247)
  (BUTTONSET TITLE "&OK;&Abandon;&Prcdent" AT 12 216 DEFAULT 3 HORZ)
  (DROPDOWN TITLE "DropDown 1;DropDown 2;DropDown 3;" TO "VariableDD" AT 108 12 WIDTH 155 HEIGHT 20 DEFAULT 2)
  (TEXT TITLE "StaticText :" AT 12 16 WIDTH 74 HEIGHT 20 CENTER)
  (CHECKBOX TITLE "Checkbox A" TO "VariableA" AT 168 120)
  (CHECKBOX TITLE "Checkbox B" TO "VariableB" AT 168 144 CHECKED)
  (RADIOBUTTON TITLE "Radiobutton 1;Radiobutton 2;Radiobutton 3" TO "Variable123" AT 12 120 DEFAULT 3)
  (EDIT TO "VariableEdit2" AT 12 72 DEFAULT "Edit 2")

 * \endcode

 * \note Un descriptif peut contenir autant de botes de dialogue DIALOG que ncessaire,
 * soit autant de lignes que de botes de dialogue et une seule ligne par dialogue.

Ce descriptif produit la bote de dialogue suivante :

 * \image html MinD_big.png

Si les valeurs ne sont pas modifies, ce dialogue produira le fichier de sortie suivant :

 * \code{.unparsed}

VariableEdit2 = "Edit 2"
VariableA = F
VariableB = T
VariableDD = "DropDown 2"
Variable123 = 3

 * \endcode

### Comment utiliser MinD ?

Aprs avoir rdig un fichier de dialogue, par exemple en copiant l'exemple de dialogue
ci-dessus dans un fichier nomm testScript.mind (l'extension .mind est facultative),
il suffit de lancer MinD avec le fichier en argument pour afficher le(s) dialogue(s).

- Sous Windows :

 * \code{.unparsed}

MinD testScript.mind

 * \endcode

- Sous Linux :

 * \code{.unparsed}

./MinD testScript.mind

 * \endcode

 * \image html MinD_big_GTK3.png

Si aucun argument n'est fourni  MinD lors de son lancement, par exemple s'il est lanc
directement par l'utilisateur en cliquant dessus, alors MinD affiche une bote de dialogue
permettant de choisir le fichier de dialogue  ouvrir.

Bien sr, pour prendre en compte les choix de l'utilisateur, il faut aussi grer
le fichier de sortie, nomm MinD.out et gnr dans le rpertoire du fichier de dialogue.
Ainsi, pour utiliser MinD, il faut lancer le fichier de dialogue, en mode synchrone,
puis traiter les lignes du fichier MinD.out en sortie pour assigner les valeurs choisies
aux variables adquates.
Selon le langage de script, la mthode sera diffrente, et il existe diverses solutions.
Dans les exemples suivants, MinD est lanc avec le dialogue simple,
soit la sortie suivante par dfaut :

 * \code{.unparsed}

VariableEdit = "Edit Box"

 * \endcode

#### Exemple d'utilisation dans un script batch (.bat) :

 * \code{.bat}

@ECHO OFF

START /B /WAIT MinD.exe testScript.mind
IF NOT EXIST MinD.out EXIT
FOR /F "tokens=1,2*" %%i IN (MinD.out) DO SET %%i%%j%%~k
ERASE MinD.out

ECHO %VariableEdit%
PAUSE

 * \endcode

#### Exemple d'utilisation dans un script Perl :

 * \note Dans cet exemple, une fonction MinDFunction est dfinie permettant si besoin
 * un usage multiple et rpt de MinD avec plus de facilit et de clart.

 * \code{.pl}

use strict;
use warnings;

my $VariableEdit = "";

MinDFunction("testScript.mind");

print $VariableEdit;



sub MinDFunction
  {
    my $result = system "MinD.exe " . $_[0];
    if ($result == 0)
      {
        open(FILE, "< MinD.out") or die "Aucun fichier de sortie MinD.out";
        while (defined (my $line = <FILE>))
          {
            eval "\$" . $line;
          }
        close(FILE);
        unlink("MinD.out");
      }
  }

 * \endcode

#### Exemple d'utilisation dans un script ACL (Audit Command Language) :

 * \code{.unparsed}

DELETE ALL OK

EXECUTE "MinD testScript.mind"

DELETE BATCH MinD OK
IF (RETURN_CODE = 0) DO SCRIPT "MinD.out"
DELETE "MinD.out" OK
DELETE BATCH MinD OK

PAUSE "%VariableEdit%"

 * \endcode

---------------------------------------

### O tlcharger MinD ?

Code source : <a href="./download/MinDSrc.zip"> MinDSrc.zip </a>

Exemples d'utilisation : <a href="./download/MinDTest.zip"> MinDTest.zip </a>

Excutable compil : <a href="./download/MinDExe.zip"> MinDExe.zip </a>

 * \note L'excutable 32 bits contenu dans ce zip correspond  la cible release
 * du projet Code::Blocks prsent dans l'archive du code source. Il a t compil
 * statiquement dans Code::Blocks 17.12 avec MinGW-64 8.1.0 i686-posix-dwarf.

---------------------------------------

### Mentions lgales

Copyright (C) 2017, 2019 Denis MARTIN <mind@musardises.fr>

---------------------------------------

Ce fichier fait partie de MinD.

MinD est un logiciel libre : il peut tre redistribu et/ou modifi
selon les termes de la Licence Publique Gnrale GNU telle que publie par
la Free Software Foundation, soit la version 3 de la licence ou toute
version ultrieure.

MinD est distribu dans l'espoir qu'il sera utile,
mais SANS AUCUNE GARANTIE ; sans mme la garantie implicite de
VALEUR MARCHANDE ou d'ADEQUATION AVEC UN BUT PARTICULIER. Voir la
Licence Publique Gnrale GNU pour plus de dtails.

Une copie de la Licence Publique Gnrale GNU devrait tre fournie
avec MinD. Si ce n'est pas le cas, voir <http://www.gnu.org/licenses/>.

---------------------------------------

This file is part of MinD.

MinD is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or any later version.

MinD is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with MinD.  If not, see <http://www.gnu.org/licenses/>.

*/


#ifndef MIND_GTK

#include <windows.h>
#include <commctrl.h>

#include <tchar.h>
#include <stdio.h>

#include "resource.h"

#include "bricks.h"
#include "script.h"


/**
  * \fn int WINAPI _tWinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, _TCHAR *lpCmdLine, int nShowCmd)
  * \brief Procdure principale - point d'entre du programme
  *
  * \param hInstance : instance du programme
  * \param hPrevInstance : toujours NULL
  * \param lpCmdLine : paramtres de la ligne de commande
  * \param nShowCmd : affichage demand
  *
  * \return zro si le programme se termine normalement
  * \return une valeur non nulle dans le cas contraire
  */
int WINAPI _tWinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, _TCHAR *lpCmdLine, int nShowCmd)
{
  // vite le plantage en cas d'utilisation de GetOpenFileName() sur le Bureau !!!

  OleInitialize(NULL);

  // initialisation permettant l'affichage des contrles avancs

  INITCOMMONCONTROLSEX initCC;

  initCC.dwSize = sizeof(INITCOMMONCONTROLSEX);
  initCC.dwICC = ICC_DATE_CLASSES | ICC_PROGRESS_CLASS | ICC_WIN95_CLASSES;
  InitCommonControlsEx(&initCC);

  // MinD est rattach par dfaut  la fentre mise en avant avec hwnd.
  // Cela n'affiche pas d'icne dans la barre des tches et fait se comporter MinD comme une fonction du programme appelant.
  // Toutefois, c'est bloquant dans certains logiciels appelant, notamment ACL en mode SYNC, d'un contrle de ce point ...

  HWND hwnd = GetForegroundWindow();

  int length = GetWindowTextLength(hwnd);
  _TCHAR *buffer = calloc(sizeof(_TCHAR), (length + 1) * sizeof(_TCHAR));
  if (buffer)
    {
      GetWindowText(hwnd, buffer, length + 1);
      if ((StrStrI(buffer, TEXT("Analytics"))) || (StrStrI(buffer, TEXT(".ACL"))))
        {
          hwnd = NULL;
        }
      free(buffer);
    }

  _TCHAR *line = _tcsdup(lpCmdLine);

#else

#include <string.h>
#include <gtk/gtk.h>

#include "mind_gtk.h"
#include "bricks.h"
#include "script.h"


/**
* \fn int main(int argc, char **argv)
* \brief Procdure principale : point d'entre du programme (version GTK)
*
* \param argc : nombre de paramtres de la ligne de commande
* \param argv : liste de pointeurs vers les paramtres
*
* \return zro si le programme se termine normalement
* \return une valeur non nulle dans le cas contraire
*/
int main(int argc, char **argv)
{
  // initialise GTK et charge l'icne

  gtk_init(&argc, &argv);

  GdkPixbuf *mainIcon = gdk_pixbuf_new_from_file("MinD.png", NULL);
  gtk_window_set_default_icon(GDK_PIXBUF(mainIcon));

  HWND hwnd = NULL;
  HINSTANCE hInstance = NULL;

  _TCHAR *line = NULL;

  if (argc > 1)
    {
      line = strdup(argv[1]);
    }

#endif

  int result = 1;

  // si aucun script n'est fourni dans la ligne de commande, permet de le choisir

  if ((line == NULL) || (line[0] == '\0'))
    {
      if (line)
      {
        free(line);
      }

      line = FileInChooser(hwnd, TEXT("Choisir un script MinD"),
                           TEXT("Tout fichier (*.*)\0*.*\0Fichier MinD (*.mind)\0*.mind\0"));
    }

  if ((line != NULL) && (line[0] != '\0'))
    {
      // rcupration du premier argument de la ligne de commande
      // suppos gal au nom du fichier de script  excuter

      _TCHAR scriptPath[MAX_PATH] = TEXT("");

      // traite les ventuels guillemets autour de l'argument

      if (line[0] == '\"')
        {
          if (_tcschr(&line[1], '\"'))
            {
              memcpy(scriptPath, &line[1], _tcscspn(&line[1], TEXT("\"")));
              scriptPath[_tcscspn(&line[1], TEXT("\""))] = '\0';
            }
        }
      else
        {
          _tcscpy(scriptPath, line);
        }

      result = ScriptBrowser(hInstance, hwnd, scriptPath);
    }

  if (line != NULL)
    {
      free(line);
    }

#ifndef MIND_GTK
  OleUninitialize();
#else
#endif

  return result;
}


