/**
 * \file script.c
 *
 * \brief Implmentation des fonctions pour la gestion des scripts
 * \author Denis MARTIN
 * \date 2017-2019
 */


/*

Copyright (C) 2017, 2019 Denis MARTIN

---------------------------------------

This file is part of MinD.

MinD is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or any later version.

MinD is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with MinD.  If not, see <http://www.gnu.org/licenses/>.

--------------------------------------------------------------------------

Ce fichier fait partie de MinD.

MinD est un logiciel libre : il peut tre redistribu et/ou modifi
selon les termes de la Licence Publique Gnrale GNU telle que publie par
la Free Software Foundation, soit la version 3 de la licence ou toute
version ultrieure.

MinD est distribu dans l'espoir qu'il sera utile,
mais SANS AUCUNE GARANTIE ; sans mme la garantie implicite de
VALEUR MARCHANDE ou d'ADEQUATION AVEC UN BUT PARTICULIER. Voir la
Licence Publique Gnrale GNU pour plus de dtails.

Une copie de la Licence Publique Gnrale GNU devrait tre fournie
avec MinD. Si ce n'est pas le cas, voir <http://www.gnu.org/licenses/>.

*/


#include "script.h"


/**
 * \fn int ScriptBrowser(HINSTANCE hInstance, HWND hwnd, _TCHAR scriptPath[MAX_PATH])
 * \brief Parcours du script MinD pour en extraire les commandes
 *
 * \param hwnd : handle de la fentre appelante
 * \param hInstance : instance de l'application
 * \param scriptPath : chemin du fichier contenant le script MinD
 *
 * \return zro en cas de succs et une valeur non nulle dans le cas contraire
 */
int ScriptBrowser(HINSTANCE hInstance,  HWND hwnd,  _TCHAR scriptPath[MAX_PATH])
{
  int result = 0;
  _TCHAR fileMode[0x100] = TEXT("rt");
  FILE *fileIn = NULL;

#ifdef UNICODE
  int byte = 0;

  fileIn = _tfopen(scriptPath, TEXT("rb"));

  if (fileIn)
    {
      byte = fgetc(fileIn);
      fclose(fileIn);
    }

  switch (byte)
    {
      case 0xEF :
        {
          _tcscpy(fileMode, TEXT("rt, ccs=UTF-8"));
          break;
        }

      case 0xFF :
        {
          _tcscpy(fileMode, TEXT("rt, ccs=UTF-16LE"));
          break;
        }

      default :
        {
          _tcscpy(fileMode, TEXT("rt"));
        }
    }
#endif

  fileIn = _tfopen(scriptPath, fileMode);

  if (fileIn)
    {
      _TCHAR *line = NULL;
      _TCHAR buffer[MAX_BUFFER];

      int dialogNum = 0;
      int dialogBack = 0;

      while ((line = _fgetts(buffer, sizeof(buffer) / sizeof(buffer[0]), fileIn)))
        {
          _TCHAR *string = line;
#ifndef MIND_GTK
#else
          if (g_utf8_validate(line, -1, NULL) == FALSE)
            {
              string = g_convert(line, -1, "UTF-8", "WINDOWS-1252", NULL, NULL, NULL);
              if (string)
                {
                  _tcsncpy(line, string, sizeof(buffer));
                  g_free(string);
                }

              string = line;
            }
#endif
          while (string[0])
            {
              if (_istspace(string[0]))
                {
                  string[0] = ' ';
                }
              ++ string;
            }

          while (line[0] == ' ')
            {
              ++ line;
            }

          MinDVariable_Clean(&mindVarGlobal);

          line = MinDVariable_ReplaceVariableInString(&mindVarGlobal, _tcsdup(line));

          if (line)
            {
              if (!dialogBack)
                {
                  if (!_tcsnicmp(line, TEXT("ASSIGN"), _tcslen(TEXT("ASSIGN"))))
                    {
                      result = 0;
                      MinDVariable_Declaration(&mindVarGlobal, &line[6]);
                    }

                  else if (!_tcsnicmp(line, TEXT("PAUSE"), _tcslen(TEXT("PAUSE"))))
                    {
                      result = MinDMessage(hInstance, hwnd, line);

                      if (result)
                        {
                          //if (MessageBoxYesNo(hwnd, TEXT("Voulez-vous abandonner le script ?"), TEXT("")) == IDNO)
                            {
                              result = 0;
                            }
                        }
                    }

                  else if (!_tcsnicmp(line, TEXT("ACCEPT"), _tcslen(TEXT("ACCEPT"))))
                    {
                      do
                        {
#ifndef MIND_GTK
                          result = DialogBoxParam(hInstance, TEXT("DLG_MinDDialog"), hwnd,
                                                  (DLGPROC)MinDInput_Procedure, (LPARAM) line);
#else
                          result = MinDInput_Procedure(hwnd, line);
#endif

                          if (result)
                            {
                              if (MessageBoxYesNo(hwnd, TEXT("Voulez-vous abandonner le script ?"), TEXT("")) == IDYES)
                                {
                                  break;
                                }
                            }
                        }
                      while (result);
                    }
                }

              if (!_tcsnicmp(line, TEXT("DIALOG"), _tcslen(TEXT("DIALOG"))))
                {
                  ++ dialogNum;

                  if (!dialogBack || (dialogBack == dialogNum))
                    {
#ifndef MIND_GTK
                      result = DialogBoxParam(hInstance, TEXT("DLG_MinDDialog"), hwnd,
                                              (DLGPROC)MinDDialog_Procedure, (LPARAM) &line[_tcslen(TEXT("DIALOG"))]);
#else
                      result = MinDDialog_Procedure(hwnd, &line[_tcslen(TEXT("DIALOG"))]);
#endif

                      dialogBack = 0;

                      switch (result)
                        {
                          case 1 : // Abandon
                            {
                              if (MessageBoxYesNo(hwnd, TEXT("Voulez-vous abandonner le script ?"), TEXT("")) == IDNO)
                                {
                                  dialogBack = dialogNum;

                                  dialogNum = 0;
                                  fseeko(fileIn, 0, SEEK_SET);
                                  result = 0;
                                }

                              break;
                            }

                          case 2 : // Prcdent
                            {
                              dialogBack = (dialogNum == 1) ? 1 : dialogNum - 1;

                              dialogNum = 0;
                              fseeko(fileIn, 0, SEEK_SET);
                              result = 0;

                              break;
                            }

                          default :
                            {
                              break;
                            }
                        }
                    }
                }

              free(line);

              if (result)
                {
                  break;
                }
            }
        }

      fclose(fileIn);
    }

  if (!result)
    {
      result = -1;

      _TCHAR *resultPath = _tcsrchr(scriptPath, c_barre_slash);

      if (resultPath)
        {
          _tcscpy(&resultPath[1], TEXT("MinD.out"));
          resultPath = scriptPath;
        }
      else
        {
          resultPath = TEXT("MinD.out");
        }

      fileMode[0] = 'w';
      FILE *fileOut = _tfopen(resultPath, fileMode);

      if (fileOut)
        {
          result = 0;

          MinDVariable_FreeAll(&mindVarGlobal, fileOut);
          fclose(fileOut);
        }
    }

  return result;
}

