/**
 * \file variable.c
 *
 * \brief Implmentation des fonctions et macros pour la gestion des variables
 * \author Denis MARTIN
 * \date 2017-2019
 */


/*

Copyright (C) 2017, 2019 Denis MARTIN

---------------------------------------

This file is part of MinD.

MinD is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or any later version.

MinD is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with MinD.  If not, see <http://www.gnu.org/licenses/>.

--------------------------------------------------------------------------

Ce fichier fait partie de MinD.

MinD est un logiciel libre : il peut tre redistribu et/ou modifi
selon les termes de la Licence Publique Gnrale GNU telle que publie par
la Free Software Foundation, soit la version 3 de la licence ou toute
version ultrieure.

MinD est distribu dans l'espoir qu'il sera utile,
mais SANS AUCUNE GARANTIE ; sans mme la garantie implicite de
VALEUR MARCHANDE ou d'ADEQUATION AVEC UN BUT PARTICULIER. Voir la
Licence Publique Gnrale GNU pour plus de dtails.

Une copie de la Licence Publique Gnrale GNU devrait tre fournie
avec MinD. Si ce n'est pas le cas, voir <http://www.gnu.org/licenses/>.

*/


#include "variable.h"


/**
 * \fn MinDVar *MinDVariable_Fill(MinDVar **mindVarPtr, int type, _TCHAR *name, int id, _TCHAR *value)
 * \brief remplissage d'une variable
 *
 * Cette fonction recherche une variable par son nom ou son ID dans la liste des
 * variables dont un pointeur est fourni. Si la variable n'existe pas, elle est cre,
 * puis sa valeur est mise  jour.
 *
 * \param mindVarPtr : adresse du pointeur sur une variable de la liste
 * \param type : type de variable (L/C/N)
 * \param name : nom de la variable
 * \param id : identifiant du contrle li  cette variable
 * \param value : valeur de la variable
 *
 * \return un pointeur sur la structure de la nouvelle variable
 * \return NULL si la variable n'a pu tre cre ou complte
 */
MinDVar *MinDVariable_Fill(MinDVar **mindVarPtr, int type, _TCHAR *name, int id, _TCHAR *value)
{
  MinDVar *result = NULL;

  if (!mindVarPtr)
    {
      mindVarPtr = &mindVarGlobal;
    }

  MinDVar *mindVar = mindVarPtr[0];

  if (name || id)
    {
      // rembobinage de la liste des variables

      while ((mindVar) && (mindVar->previous))
        {
          mindVar = mindVar->previous;
        }

      mindVarPtr[0] = mindVar;
      MinDVar *mindVarPrevious = NULL;

      // recherche de la variable

      while (mindVar)
        {
          if (((name) && (mindVar->name) && (!_tcsicmp(mindVar->name, name)))
              || ((id) && (mindVar->id) && (id == mindVar->id)))
            {
              // OK, la variable existe -> mise  jour

              result = mindVar;

              mindVar->type = _totupper(type);
              if ((mindVar->type != 'L') && (mindVar->type != 'N'))
                {
                  mindVar->type = 'C';
                }

              if (name)
                {
                  if (mindVar->name)
                    {
                      free(mindVar->name);
                    }

                  mindVar->name = _tcsdup(name);
                }

              if (id)
                {
                  mindVar->id = id;
                }

              if (value)
                {
                  if (mindVar->value)
                    {
                      free(mindVar->value);
                    }

                  mindVar->value = _tcsdup(value);
                }

              break;
            }

          mindVarPrevious = mindVar;
          mindVar = mindVar->next;
        }

      if (!mindVar)
        {
          mindVar = calloc(sizeof(_TCHAR), sizeof(MinDVar));
          result = mindVar;
          if (mindVar)
            {
              if (mindVarPrevious)
                {
                  mindVarPrevious->next = mindVar;
                }

              mindVar->type = _totupper(type);
              if ((mindVar->type != 'L') && (mindVar->type != 'N'))
                {
                  mindVar->type = 'C';
                }

              mindVar->name = _tcsdup(name);
              mindVar->id = id;
              if (value)
                {
                  mindVar->value = _tcsdup(value);
                }
              mindVar->previous = mindVarPrevious;
              mindVar->next = NULL;

              if (!mindVarPtr[0])
                {
                  mindVarPtr[0] = mindVar;
                }
            }
        }
    }

  return result;
}


/**
 * \fn MinDVar *MinDVariable_Retrieve(MinDVar **mindVarPtr, _TCHAR *name, int id)
 * \brief rcupre une variable
 *
 * Cette fonction recherche une variable par son nom ou son ID dans la liste des
 * variables dont un pointeur est fourni.
 *
 * \param mindVarPtr : adresse du pointeur sur une variable de la liste
 * \param name : nom de la variable
 * \param id : identifiant du contrle li  cette variable
 *
 * \return un pointeur sur la structure de la variable
 * \return NULL si la variable n'a pu tre trouve
 */
MinDVar *MinDVariable_Retrieve(MinDVar **mindVarPtr, _TCHAR *name, int id)
{
  MinDVar *result = NULL;

  if (!mindVarPtr)
    {
      mindVarPtr = &mindVarGlobal;
    }

  MinDVar *mindVar = mindVarPtr[0];

  if (name || id)
    {
      // rembobinage de la liste des variables

      while ((mindVar) && (mindVar->previous))
        {
          mindVar = mindVar->previous;
        }

      mindVarPtr[0] = mindVar;

      // recherche de la variable

      while (mindVar)
        {
          if (((name) && (mindVar->name) && (!_tcsicmp(mindVar->name, name)))
              || ((id) && (mindVar->id) && (id == mindVar->id)))
            {
              // OK, la variable existe

              result = mindVar;
              break;
            }

          mindVar = mindVar->next;
        }
    }

  return result;
}


/**
 * \fn int MinDVariable_FreeAll(MinDVar **mindVarPtr, FILE *fileOut)
 * \brief inscrit dans un fichier de sortie toutes les variables d'une liste et les libre
 *
 * \param mindVarPtr : adresse du pointeur sur une variable de la liste
 * \param fileOut : pointeur vers le descripteur du fichier de sortie
 *
 * \return le nombre de variables libres
 */
int MinDVariable_FreeAll(MinDVar **mindVarPtr, FILE *fileOut)
{
  int result = 0;

  if (!mindVarPtr)
    {
      mindVarPtr = &mindVarGlobal;
    }

  MinDVar *mindVar = mindVarPtr[0];

  // rembobinage de la liste des variables

  while ((mindVar) && (mindVar->previous))
    {
      mindVar = mindVar->previous;
    }

  // parcours de la liste

  while (mindVar)
    {
      ++ result;

      if (fileOut && mindVar->name && mindVar->name[0])
        {
          switch (mindVar->type)
            {
              case 'L' :
                {
                  int value = 'F';
                  if (mindVar->value && mindVar->value[0])
                    {
                      value = _totupper(mindVar->value[0]);
                    }

                  _ftprintf(fileOut, TEXT("%s = %s\n"), mindVar->name,
                          ((value == 'F') || (value == '0') || (value == 'N')) ? TEXT("F") : TEXT("T"));

                  break;
                }

              case 'N' :
                {
                  long value = 0;
                  if (mindVar->value && mindVar->value[0])
                    {
                      value = _tcstol(mindVar->value, NULL, 10);
                    }

                  _ftprintf(fileOut, TEXT("%s = %ld\n"), mindVar->name, value);

                  break;
                }

              default :
                {
                  // remplacement des ventuels " par des ' dans la chane

                  _TCHAR *value = mindVar->value ? mindVar->value : TEXT("");
                  _TCHAR *ptr = NULL;
                  while ((ptr = _tcschr(value, '\"')))
                    {
                      ptr[0] = '\'';
                    }

                  _ftprintf(fileOut, TEXT("%s = \"%s\"\n"), mindVar->name, value);
                }
            }
        }

      if (mindVar->name)
        {
          free(mindVar->name);
        }

      if (mindVar->value)
        {
          free(mindVar->value);
        }

      MinDVar *mindVarPrevious = mindVar;
      mindVar = mindVar->next;
      free(mindVarPrevious);
    }

  mindVarPtr[0] = mindVar;

  return result;
}


/**
 * \fn int MinDVariable_Clean(MinDVar **mindVarPtr)
 * \brief met  zro les ID de toutes les variables d'une liste
 *
 * \param mindVarPtr : adresse du pointeur sur une variable de la liste
 *
 * \return le nombre de variables nettoyes
 */
int MinDVariable_Clean(MinDVar **mindVarPtr)
{
  int result = 0;

  if (!mindVarPtr)
    {
      mindVarPtr = &mindVarGlobal;
    }

  MinDVar *mindVar = mindVarPtr[0];

  // rembobinage de la liste des variables

  while ((mindVar) && (mindVar->previous))
    {
      mindVar = mindVar->previous;
    }

  mindVarPtr[0] = mindVar;

  // parcours de la liste

  while (mindVar)
    {
      ++ result;

      mindVar->id = 0;
      mindVar = mindVar->next;
    }

  return result;
}


/**
 * \fn _TCHAR *MinDVariable_ReplaceVariableInString(MinDVar **mindVarPtr, _TCHAR *text)
 * \brief remplace dans une chane le nom d'une variable (entoure de %) par son contenu
 * \warning la fonction peut modifier ou supprimer la chane d'origine !
 *
 * \param mindVarPtr : adresse du pointeur sur une variable de la liste
 * \param text : chane d'origine
 *
 * \return un pointeur vers la nouvelle chane
 */
_TCHAR *MinDVariable_ReplaceVariableInString(MinDVar **mindVarPtr, _TCHAR *text)
{
  _TCHAR *result = NULL;

  if (!mindVarPtr)
    {
      mindVarPtr = &mindVarGlobal;
    }

  if (text && text[0])
    {
      _TCHAR *variable = text;
      while ((variable = _tcschr(variable, '%')))
        {
          variable[0] = '\0';

          _TCHAR *variableEnd = _tcschr(variable + 1, '%');
          if (variableEnd)
            {
              variableEnd[0] = '\0';
              MinDVar *mindVar = MinDVariable_Retrieve(mindVarPtr, variable + 1, 0);

              if (mindVar)
                {
                  int length = _tcslen(text) + _tcslen(variableEnd + 1) + (mindVar->value ? _tcslen(mindVar->value) : 0);
                  result = calloc(sizeof(_TCHAR), length + 1);
                  if (result)
                    {
                      _tcscpy(result, text);
                      _tcscat(result, mindVar->value);
                      _tcscat(result, variableEnd + 1);
                    }
                }

              variableEnd[0] = '%';
            }

          if (result)
            {
              variable = result + (variable - text);
              free(text);
              text = result;
              result = NULL;
            }
          else
            {
              variable[0] = '%';
            }

          ++ variable;
        }
    }

  result = text;

  return result;
}


/**
 * \fn MinDVar *MinDVariable_Declaration(MinDVar **mindVarPtr, _TCHAR *line)
 * \brief prise en compte de la dclaration d'une variable
 *
 * \param mindVarPtr : adresse du pointeur sur une variable de la liste
 * \param line : ligne de commande (peut tre modifie par la fonction !)
 *
 * \return un pointeur sur la structure de la variable
 * \return NULL si la variable n'a pu tre trouve
 */
MinDVar *MinDVariable_Declaration(MinDVar **mindVarPtr, _TCHAR *line)
{
  MinDVar *result = NULL;

  if (!mindVarPtr)
    {
      mindVarPtr = &mindVarGlobal;
    }

  if (line)
    {
      while (line[0] == ' ')
        {
          ++ line;
        }

      _TCHAR *variable = line;

      while (line[0] && line[0] != ' ')
        {
          ++ line;
        }

      if (line [0])
        {
          line[0] = '\0';
          ++ line;

          while (line[0] && line[0] != '=')
            {
              ++ line;
            }

          if (line [0])
            {
              ++ line;
              while (line[0] == ' ')
                {
                  ++ line;
                }

              int quote = 0;
              if ((line[0] == '\'') || (line[0] == '\"'))
                {
                  quote = line[0];
                  ++ line;
                }

              _TCHAR *value = line;

              while (line[0] && ((!quote && (line[0] != ' ')) || (quote && (line[0] != quote))))
                {
                  ++ line;
                }

              line[0] = '\0';

              if (variable[0])
                {
                  int type = 'C';

                  if (!quote)
                    {
                      if (_tcsspn(value, TEXT("-+.,0123456789")) == _tcslen(value))
                        {
                          type = 'N';
                        }
                      else if (!_tcsicmp(value, TEXT("TRUE")) || !_tcsicmp(value, TEXT("T")))
                        {
                          type = 'L';
                          value = TEXT("TRUE");
                        }
                      else if (!_tcsicmp(value, TEXT("FALSE")) || !_tcsicmp(value, TEXT("F")))
                        {
                          type = 'L';
                          value = TEXT("FALSE");
                        }
                    }

                  result = MinDVariable_Fill(mindVarPtr, type, variable, 0, value);
                }
            }
        }
    }

  return result;
}

